{-# LANGUAGE ImplicitParams, RankNTypes, TypeApplications #-}


-- | Copyright  : Will Thompson and Iñaki García Etxebarria
-- License    : LGPL-2.1
-- Maintainer : Iñaki García Etxebarria
-- 
-- Dialogs are a convenient way to prompt the user for a small amount
-- of input.
-- 
-- \<picture>
--   \<source srcset=\"dialog-dark.png\" media=\"(prefers-color-scheme: dark)\">
--   \<img alt=\"An example GtkDialog\" src=\"dialog.png\">
-- \<\/picture>
-- 
-- Typical uses are to display a message, ask a question, or anything else
-- that does not require extensive effort on the user’s part.
-- 
-- The main area of a @GtkDialog@ is called the \"content area\", and is yours
-- to populate with widgets such a @GtkLabel@ or @GtkEntry@, to present
-- your information, questions, or tasks to the user.
-- 
-- In addition, dialogs allow you to add \"action widgets\". Most commonly,
-- action widgets are buttons. Depending on the platform, action widgets may
-- be presented in the header bar at the top of the window, or at the bottom
-- of the window. To add action widgets, create your @GtkDialog@ using
-- t'GI.Gtk.Objects.Dialog.Dialog'.@/new_with_buttons/@(), or use
-- 'GI.Gtk.Objects.Dialog.dialogAddButton', t'GI.Gtk.Objects.Dialog.Dialog'.@/add_buttons/@(),
-- or 'GI.Gtk.Objects.Dialog.dialogAddActionWidget'.
-- 
-- @GtkDialogs@ uses some heuristics to decide whether to add a close
-- button to the window decorations. If any of the action buttons use
-- the response ID 'GI.Gtk.Enums.ResponseTypeClose' or 'GI.Gtk.Enums.ResponseTypeCancel', the
-- close button is omitted.
-- 
-- Clicking a button that was added as an action widget will emit the
-- [Dialog::response]("GI.Gtk.Objects.Dialog#g:signal:response") signal with a response ID that you specified.
-- GTK will never assign a meaning to positive response IDs; these are
-- entirely user-defined. But for convenience, you can use the response
-- IDs in the t'GI.Gtk.Enums.ResponseType' enumeration (these all have values
-- less than zero). If a dialog receives a delete event, the
-- [Dialog::response]("GI.Gtk.Objects.Dialog#g:signal:response") signal will be emitted with the
-- 'GI.Gtk.Enums.ResponseTypeDeleteEvent' response ID.
-- 
-- Dialogs are created with a call to 'GI.Gtk.Objects.Dialog.dialogNew' or
-- t'GI.Gtk.Objects.Dialog.Dialog'.@/new_with_buttons/@(). The latter is recommended; it allows
-- you to set the dialog title, some convenient flags, and add buttons.
-- 
-- A “modal” dialog (that is, one which freezes the rest of the application
-- from user input), can be created by calling 'GI.Gtk.Objects.Window.windowSetModal'
-- on the dialog. When using t'GI.Gtk.Objects.Dialog.Dialog'.@/new_with_buttons/@(), you can also
-- pass the 'GI.Gtk.Flags.DialogFlagsModal' flag to make a dialog modal.
-- 
-- For the simple dialog in the following example, a t'GI.Gtk.Objects.MessageDialog.MessageDialog'
-- would save some effort. But you’d need to create the dialog contents manually
-- if you had more than a simple message in the dialog.
-- 
-- An example for simple @GtkDialog@ usage:
-- 
-- 
-- === /c code/
-- >// Function to open a dialog box with a message
-- >void
-- >quick_message (GtkWindow *parent, char *message)
-- >{
-- > GtkWidget *dialog, *label, *content_area;
-- > GtkDialogFlags flags;
-- >
-- > // Create the widgets
-- > flags = GTK_DIALOG_DESTROY_WITH_PARENT;
-- > dialog = gtk_dialog_new_with_buttons ("Message",
-- >                                       parent,
-- >                                       flags,
-- >                                       _("_OK"),
-- >                                       GTK_RESPONSE_NONE,
-- >                                       NULL);
-- > content_area = gtk_dialog_get_content_area (GTK_DIALOG (dialog));
-- > label = gtk_label_new (message);
-- >
-- > // Ensure that the dialog box is destroyed when the user responds
-- >
-- > g_signal_connect_swapped (dialog,
-- >                           "response",
-- >                           G_CALLBACK (gtk_window_destroy),
-- >                           dialog);
-- >
-- > // Add the label, and show everything we’ve added
-- >
-- > gtk_box_append (GTK_BOX (content_area), label);
-- > gtk_widget_show (dialog);
-- >}
-- 
-- 
-- = GtkDialog as GtkBuildable
-- 
-- The @GtkDialog@ implementation of the @GtkBuildable@ interface exposes the
-- /@contentArea@/ as an internal child with the name “content_area”.
-- 
-- @GtkDialog@ supports a custom @\<action-widgets>@ element, which can contain
-- multiple @\<action-widget>@ elements. The “response” attribute specifies a
-- numeric response, and the content of the element is the id of widget
-- (which should be a child of the dialogs /@actionArea@/). To mark a response
-- as default, set the “default” attribute of the @\<action-widget>@ element
-- to true.
-- 
-- @GtkDialog@ supports adding action widgets by specifying “action” as
-- the “type” attribute of a @\<child>@ element. The widget will be added
-- either to the action area or the headerbar of the dialog, depending
-- on the “use-header-bar” property. The response id has to be associated
-- with the action widget using the @\<action-widgets>@ element.
-- 
-- An example of a @GtkDialog@ UI definition fragment:
-- 
-- 
-- === /xml code/
-- ><object class="GtkDialog" id="dialog1">
-- >  <child type="action">
-- >    <object class="GtkButton" id="button_cancel"/>
-- >  </child>
-- >  <child type="action">
-- >    <object class="GtkButton" id="button_ok">
-- >    </object>
-- >  </child>
-- >  <action-widgets>
-- >    <action-widget response="cancel">button_cancel</action-widget>
-- >    <action-widget response="ok" default="true">button_ok</action-widget>
-- >  </action-widgets>
-- ></object>
-- 
-- 
-- = Accessibility
-- 
-- @GtkDialog@ uses the 'GI.Gtk.Enums.AccessibleRoleDialog' role.

#if (MIN_VERSION_haskell_gi_overloading(1,0,0) && !defined(__HADDOCK_VERSION__))
#define ENABLE_OVERLOADING
#endif

module GI.Gtk.Objects.Dialog
    ( 

-- * Exported types
    Dialog(..)                              ,
    IsDialog                                ,
    toDialog                                ,


 -- * Methods
-- | 
-- 
--  === __Click to display all available methods, including inherited ones__
-- ==== Methods
-- [actionSetEnabled]("GI.Gtk.Objects.Widget#g:method:actionSetEnabled"), [activate]("GI.Gtk.Objects.Widget#g:method:activate"), [activateAction]("GI.Gtk.Objects.Widget#g:method:activateAction"), [activateDefault]("GI.Gtk.Objects.Widget#g:method:activateDefault"), [addActionWidget]("GI.Gtk.Objects.Dialog#g:method:addActionWidget"), [addButton]("GI.Gtk.Objects.Dialog#g:method:addButton"), [addController]("GI.Gtk.Objects.Widget#g:method:addController"), [addCssClass]("GI.Gtk.Objects.Widget#g:method:addCssClass"), [addMnemonicLabel]("GI.Gtk.Objects.Widget#g:method:addMnemonicLabel"), [addTickCallback]("GI.Gtk.Objects.Widget#g:method:addTickCallback"), [allocate]("GI.Gtk.Objects.Widget#g:method:allocate"), [announce]("GI.Gtk.Interfaces.Accessible#g:method:announce"), [bindProperty]("GI.GObject.Objects.Object#g:method:bindProperty"), [bindPropertyFull]("GI.GObject.Objects.Object#g:method:bindPropertyFull"), [childFocus]("GI.Gtk.Objects.Widget#g:method:childFocus"), [close]("GI.Gtk.Objects.Window#g:method:close"), [computeBounds]("GI.Gtk.Objects.Widget#g:method:computeBounds"), [computeExpand]("GI.Gtk.Objects.Widget#g:method:computeExpand"), [computePoint]("GI.Gtk.Objects.Widget#g:method:computePoint"), [computeTransform]("GI.Gtk.Objects.Widget#g:method:computeTransform"), [contains]("GI.Gtk.Objects.Widget#g:method:contains"), [createPangoContext]("GI.Gtk.Objects.Widget#g:method:createPangoContext"), [createPangoLayout]("GI.Gtk.Objects.Widget#g:method:createPangoLayout"), [destroy]("GI.Gtk.Objects.Window#g:method:destroy"), [disposeTemplate]("GI.Gtk.Objects.Widget#g:method:disposeTemplate"), [dragCheckThreshold]("GI.Gtk.Objects.Widget#g:method:dragCheckThreshold"), [errorBell]("GI.Gtk.Objects.Widget#g:method:errorBell"), [forceFloating]("GI.GObject.Objects.Object#g:method:forceFloating"), [freezeNotify]("GI.GObject.Objects.Object#g:method:freezeNotify"), [fullscreen]("GI.Gtk.Objects.Window#g:method:fullscreen"), [fullscreenOnMonitor]("GI.Gtk.Objects.Window#g:method:fullscreenOnMonitor"), [getv]("GI.GObject.Objects.Object#g:method:getv"), [grabFocus]("GI.Gtk.Objects.Widget#g:method:grabFocus"), [hasCssClass]("GI.Gtk.Objects.Widget#g:method:hasCssClass"), [hasDefault]("GI.Gtk.Objects.Widget#g:method:hasDefault"), [hasFocus]("GI.Gtk.Objects.Widget#g:method:hasFocus"), [hasGroup]("GI.Gtk.Objects.Window#g:method:hasGroup"), [hasVisibleFocus]("GI.Gtk.Objects.Widget#g:method:hasVisibleFocus"), [hide]("GI.Gtk.Objects.Widget#g:method:hide"), [inDestruction]("GI.Gtk.Objects.Widget#g:method:inDestruction"), [initTemplate]("GI.Gtk.Objects.Widget#g:method:initTemplate"), [insertActionGroup]("GI.Gtk.Objects.Widget#g:method:insertActionGroup"), [insertAfter]("GI.Gtk.Objects.Widget#g:method:insertAfter"), [insertBefore]("GI.Gtk.Objects.Widget#g:method:insertBefore"), [isActive]("GI.Gtk.Objects.Window#g:method:isActive"), [isAncestor]("GI.Gtk.Objects.Widget#g:method:isAncestor"), [isDrawable]("GI.Gtk.Objects.Widget#g:method:isDrawable"), [isFloating]("GI.GObject.Objects.Object#g:method:isFloating"), [isFocus]("GI.Gtk.Objects.Widget#g:method:isFocus"), [isFullscreen]("GI.Gtk.Objects.Window#g:method:isFullscreen"), [isMaximized]("GI.Gtk.Objects.Window#g:method:isMaximized"), [isSensitive]("GI.Gtk.Objects.Widget#g:method:isSensitive"), [isSuspended]("GI.Gtk.Objects.Window#g:method:isSuspended"), [isVisible]("GI.Gtk.Objects.Widget#g:method:isVisible"), [keynavFailed]("GI.Gtk.Objects.Widget#g:method:keynavFailed"), [listMnemonicLabels]("GI.Gtk.Objects.Widget#g:method:listMnemonicLabels"), [map]("GI.Gtk.Objects.Widget#g:method:map"), [maximize]("GI.Gtk.Objects.Window#g:method:maximize"), [measure]("GI.Gtk.Objects.Widget#g:method:measure"), [minimize]("GI.Gtk.Objects.Window#g:method:minimize"), [mnemonicActivate]("GI.Gtk.Objects.Widget#g:method:mnemonicActivate"), [notify]("GI.GObject.Objects.Object#g:method:notify"), [notifyByPspec]("GI.GObject.Objects.Object#g:method:notifyByPspec"), [observeChildren]("GI.Gtk.Objects.Widget#g:method:observeChildren"), [observeControllers]("GI.Gtk.Objects.Widget#g:method:observeControllers"), [pick]("GI.Gtk.Objects.Widget#g:method:pick"), [present]("GI.Gtk.Objects.Window#g:method:present"), [presentWithTime]("GI.Gtk.Objects.Window#g:method:presentWithTime"), [queueAllocate]("GI.Gtk.Objects.Widget#g:method:queueAllocate"), [queueDraw]("GI.Gtk.Objects.Widget#g:method:queueDraw"), [queueResize]("GI.Gtk.Objects.Widget#g:method:queueResize"), [realize]("GI.Gtk.Objects.Widget#g:method:realize"), [ref]("GI.GObject.Objects.Object#g:method:ref"), [refSink]("GI.GObject.Objects.Object#g:method:refSink"), [removeController]("GI.Gtk.Objects.Widget#g:method:removeController"), [removeCssClass]("GI.Gtk.Objects.Widget#g:method:removeCssClass"), [removeMnemonicLabel]("GI.Gtk.Objects.Widget#g:method:removeMnemonicLabel"), [removeTickCallback]("GI.Gtk.Objects.Widget#g:method:removeTickCallback"), [resetProperty]("GI.Gtk.Interfaces.Accessible#g:method:resetProperty"), [resetRelation]("GI.Gtk.Interfaces.Accessible#g:method:resetRelation"), [resetState]("GI.Gtk.Interfaces.Accessible#g:method:resetState"), [response]("GI.Gtk.Objects.Dialog#g:method:response"), [runDispose]("GI.GObject.Objects.Object#g:method:runDispose"), [shouldLayout]("GI.Gtk.Objects.Widget#g:method:shouldLayout"), [show]("GI.Gtk.Objects.Widget#g:method:show"), [sizeAllocate]("GI.Gtk.Objects.Widget#g:method:sizeAllocate"), [snapshotChild]("GI.Gtk.Objects.Widget#g:method:snapshotChild"), [stealData]("GI.GObject.Objects.Object#g:method:stealData"), [stealQdata]("GI.GObject.Objects.Object#g:method:stealQdata"), [thawNotify]("GI.GObject.Objects.Object#g:method:thawNotify"), [translateCoordinates]("GI.Gtk.Objects.Widget#g:method:translateCoordinates"), [triggerTooltipQuery]("GI.Gtk.Objects.Widget#g:method:triggerTooltipQuery"), [unfullscreen]("GI.Gtk.Objects.Window#g:method:unfullscreen"), [unmap]("GI.Gtk.Objects.Widget#g:method:unmap"), [unmaximize]("GI.Gtk.Objects.Window#g:method:unmaximize"), [unminimize]("GI.Gtk.Objects.Window#g:method:unminimize"), [unparent]("GI.Gtk.Objects.Widget#g:method:unparent"), [unrealize]("GI.Gtk.Objects.Widget#g:method:unrealize"), [unref]("GI.GObject.Objects.Object#g:method:unref"), [unsetStateFlags]("GI.Gtk.Objects.Widget#g:method:unsetStateFlags"), [updateNextAccessibleSibling]("GI.Gtk.Interfaces.Accessible#g:method:updateNextAccessibleSibling"), [updatePlatformState]("GI.Gtk.Interfaces.Accessible#g:method:updatePlatformState"), [updateProperty]("GI.Gtk.Interfaces.Accessible#g:method:updateProperty"), [updateRelation]("GI.Gtk.Interfaces.Accessible#g:method:updateRelation"), [updateState]("GI.Gtk.Interfaces.Accessible#g:method:updateState"), [watchClosure]("GI.GObject.Objects.Object#g:method:watchClosure").
-- 
-- ==== Getters
-- [getAccessibleParent]("GI.Gtk.Interfaces.Accessible#g:method:getAccessibleParent"), [getAccessibleRole]("GI.Gtk.Interfaces.Accessible#g:method:getAccessibleRole"), [getAllocatedBaseline]("GI.Gtk.Objects.Widget#g:method:getAllocatedBaseline"), [getAllocatedHeight]("GI.Gtk.Objects.Widget#g:method:getAllocatedHeight"), [getAllocatedWidth]("GI.Gtk.Objects.Widget#g:method:getAllocatedWidth"), [getAllocation]("GI.Gtk.Objects.Widget#g:method:getAllocation"), [getAncestor]("GI.Gtk.Objects.Widget#g:method:getAncestor"), [getApplication]("GI.Gtk.Objects.Window#g:method:getApplication"), [getAtContext]("GI.Gtk.Interfaces.Accessible#g:method:getAtContext"), [getBaseline]("GI.Gtk.Objects.Widget#g:method:getBaseline"), [getBounds]("GI.Gtk.Interfaces.Accessible#g:method:getBounds"), [getBuildableId]("GI.Gtk.Interfaces.Buildable#g:method:getBuildableId"), [getCanFocus]("GI.Gtk.Objects.Widget#g:method:getCanFocus"), [getCanTarget]("GI.Gtk.Objects.Widget#g:method:getCanTarget"), [getChild]("GI.Gtk.Objects.Window#g:method:getChild"), [getChildVisible]("GI.Gtk.Objects.Widget#g:method:getChildVisible"), [getClipboard]("GI.Gtk.Objects.Widget#g:method:getClipboard"), [getColor]("GI.Gtk.Objects.Widget#g:method:getColor"), [getContentArea]("GI.Gtk.Objects.Dialog#g:method:getContentArea"), [getCssClasses]("GI.Gtk.Objects.Widget#g:method:getCssClasses"), [getCssName]("GI.Gtk.Objects.Widget#g:method:getCssName"), [getCursor]("GI.Gtk.Objects.Widget#g:method:getCursor"), [getData]("GI.GObject.Objects.Object#g:method:getData"), [getDecorated]("GI.Gtk.Objects.Window#g:method:getDecorated"), [getDefaultSize]("GI.Gtk.Objects.Window#g:method:getDefaultSize"), [getDefaultWidget]("GI.Gtk.Objects.Window#g:method:getDefaultWidget"), [getDeletable]("GI.Gtk.Objects.Window#g:method:getDeletable"), [getDestroyWithParent]("GI.Gtk.Objects.Window#g:method:getDestroyWithParent"), [getDirection]("GI.Gtk.Objects.Widget#g:method:getDirection"), [getDisplay]("GI.Gtk.Objects.Widget#g:method:getDisplay"), [getFirstAccessibleChild]("GI.Gtk.Interfaces.Accessible#g:method:getFirstAccessibleChild"), [getFirstChild]("GI.Gtk.Objects.Widget#g:method:getFirstChild"), [getFocus]("GI.Gtk.Objects.Window#g:method:getFocus"), [getFocusChild]("GI.Gtk.Objects.Widget#g:method:getFocusChild"), [getFocusOnClick]("GI.Gtk.Objects.Widget#g:method:getFocusOnClick"), [getFocusVisible]("GI.Gtk.Objects.Window#g:method:getFocusVisible"), [getFocusable]("GI.Gtk.Objects.Widget#g:method:getFocusable"), [getFontMap]("GI.Gtk.Objects.Widget#g:method:getFontMap"), [getFontOptions]("GI.Gtk.Objects.Widget#g:method:getFontOptions"), [getFrameClock]("GI.Gtk.Objects.Widget#g:method:getFrameClock"), [getGravity]("GI.Gtk.Objects.Window#g:method:getGravity"), [getGroup]("GI.Gtk.Objects.Window#g:method:getGroup"), [getHalign]("GI.Gtk.Objects.Widget#g:method:getHalign"), [getHandleMenubarAccel]("GI.Gtk.Objects.Window#g:method:getHandleMenubarAccel"), [getHasTooltip]("GI.Gtk.Objects.Widget#g:method:getHasTooltip"), [getHeaderBar]("GI.Gtk.Objects.Dialog#g:method:getHeaderBar"), [getHeight]("GI.Gtk.Objects.Widget#g:method:getHeight"), [getHexpand]("GI.Gtk.Objects.Widget#g:method:getHexpand"), [getHexpandSet]("GI.Gtk.Objects.Widget#g:method:getHexpandSet"), [getHideOnClose]("GI.Gtk.Objects.Window#g:method:getHideOnClose"), [getIconName]("GI.Gtk.Objects.Window#g:method:getIconName"), [getLastChild]("GI.Gtk.Objects.Widget#g:method:getLastChild"), [getLayoutManager]("GI.Gtk.Objects.Widget#g:method:getLayoutManager"), [getLimitEvents]("GI.Gtk.Objects.Widget#g:method:getLimitEvents"), [getMapped]("GI.Gtk.Objects.Widget#g:method:getMapped"), [getMarginBottom]("GI.Gtk.Objects.Widget#g:method:getMarginBottom"), [getMarginEnd]("GI.Gtk.Objects.Widget#g:method:getMarginEnd"), [getMarginStart]("GI.Gtk.Objects.Widget#g:method:getMarginStart"), [getMarginTop]("GI.Gtk.Objects.Widget#g:method:getMarginTop"), [getMnemonicsVisible]("GI.Gtk.Objects.Window#g:method:getMnemonicsVisible"), [getModal]("GI.Gtk.Objects.Window#g:method:getModal"), [getName]("GI.Gtk.Objects.Widget#g:method:getName"), [getNative]("GI.Gtk.Objects.Widget#g:method:getNative"), [getNextAccessibleSibling]("GI.Gtk.Interfaces.Accessible#g:method:getNextAccessibleSibling"), [getNextSibling]("GI.Gtk.Objects.Widget#g:method:getNextSibling"), [getOpacity]("GI.Gtk.Objects.Widget#g:method:getOpacity"), [getOverflow]("GI.Gtk.Objects.Widget#g:method:getOverflow"), [getPangoContext]("GI.Gtk.Objects.Widget#g:method:getPangoContext"), [getParent]("GI.Gtk.Objects.Widget#g:method:getParent"), [getPlatformState]("GI.Gtk.Interfaces.Accessible#g:method:getPlatformState"), [getPreferredSize]("GI.Gtk.Objects.Widget#g:method:getPreferredSize"), [getPrevSibling]("GI.Gtk.Objects.Widget#g:method:getPrevSibling"), [getPrimaryClipboard]("GI.Gtk.Objects.Widget#g:method:getPrimaryClipboard"), [getProperty]("GI.GObject.Objects.Object#g:method:getProperty"), [getQdata]("GI.GObject.Objects.Object#g:method:getQdata"), [getRealized]("GI.Gtk.Objects.Widget#g:method:getRealized"), [getReceivesDefault]("GI.Gtk.Objects.Widget#g:method:getReceivesDefault"), [getRenderer]("GI.Gtk.Interfaces.Native#g:method:getRenderer"), [getRequestMode]("GI.Gtk.Objects.Widget#g:method:getRequestMode"), [getResizable]("GI.Gtk.Objects.Window#g:method:getResizable"), [getResponseForWidget]("GI.Gtk.Objects.Dialog#g:method:getResponseForWidget"), [getRoot]("GI.Gtk.Objects.Widget#g:method:getRoot"), [getScaleFactor]("GI.Gtk.Objects.Widget#g:method:getScaleFactor"), [getSensitive]("GI.Gtk.Objects.Widget#g:method:getSensitive"), [getSettings]("GI.Gtk.Objects.Widget#g:method:getSettings"), [getSize]("GI.Gtk.Objects.Widget#g:method:getSize"), [getSizeRequest]("GI.Gtk.Objects.Widget#g:method:getSizeRequest"), [getStateFlags]("GI.Gtk.Objects.Widget#g:method:getStateFlags"), [getStyleContext]("GI.Gtk.Objects.Widget#g:method:getStyleContext"), [getSurface]("GI.Gtk.Interfaces.Native#g:method:getSurface"), [getSurfaceTransform]("GI.Gtk.Interfaces.Native#g:method:getSurfaceTransform"), [getTemplateChild]("GI.Gtk.Objects.Widget#g:method:getTemplateChild"), [getTitle]("GI.Gtk.Objects.Window#g:method:getTitle"), [getTitlebar]("GI.Gtk.Objects.Window#g:method:getTitlebar"), [getTooltipMarkup]("GI.Gtk.Objects.Widget#g:method:getTooltipMarkup"), [getTooltipText]("GI.Gtk.Objects.Widget#g:method:getTooltipText"), [getTransientFor]("GI.Gtk.Objects.Window#g:method:getTransientFor"), [getValign]("GI.Gtk.Objects.Widget#g:method:getValign"), [getVexpand]("GI.Gtk.Objects.Widget#g:method:getVexpand"), [getVexpandSet]("GI.Gtk.Objects.Widget#g:method:getVexpandSet"), [getVisible]("GI.Gtk.Objects.Widget#g:method:getVisible"), [getWidgetForResponse]("GI.Gtk.Objects.Dialog#g:method:getWidgetForResponse"), [getWidth]("GI.Gtk.Objects.Widget#g:method:getWidth").
-- 
-- ==== Setters
-- [setAccessibleParent]("GI.Gtk.Interfaces.Accessible#g:method:setAccessibleParent"), [setApplication]("GI.Gtk.Objects.Window#g:method:setApplication"), [setCanFocus]("GI.Gtk.Objects.Widget#g:method:setCanFocus"), [setCanTarget]("GI.Gtk.Objects.Widget#g:method:setCanTarget"), [setChild]("GI.Gtk.Objects.Window#g:method:setChild"), [setChildVisible]("GI.Gtk.Objects.Widget#g:method:setChildVisible"), [setCssClasses]("GI.Gtk.Objects.Widget#g:method:setCssClasses"), [setCursor]("GI.Gtk.Objects.Widget#g:method:setCursor"), [setCursorFromName]("GI.Gtk.Objects.Widget#g:method:setCursorFromName"), [setData]("GI.GObject.Objects.Object#g:method:setData"), [setDataFull]("GI.GObject.Objects.Object#g:method:setDataFull"), [setDecorated]("GI.Gtk.Objects.Window#g:method:setDecorated"), [setDefaultResponse]("GI.Gtk.Objects.Dialog#g:method:setDefaultResponse"), [setDefaultSize]("GI.Gtk.Objects.Window#g:method:setDefaultSize"), [setDefaultWidget]("GI.Gtk.Objects.Window#g:method:setDefaultWidget"), [setDeletable]("GI.Gtk.Objects.Window#g:method:setDeletable"), [setDestroyWithParent]("GI.Gtk.Objects.Window#g:method:setDestroyWithParent"), [setDirection]("GI.Gtk.Objects.Widget#g:method:setDirection"), [setDisplay]("GI.Gtk.Objects.Window#g:method:setDisplay"), [setFocus]("GI.Gtk.Objects.Window#g:method:setFocus"), [setFocusChild]("GI.Gtk.Objects.Widget#g:method:setFocusChild"), [setFocusOnClick]("GI.Gtk.Objects.Widget#g:method:setFocusOnClick"), [setFocusVisible]("GI.Gtk.Objects.Window#g:method:setFocusVisible"), [setFocusable]("GI.Gtk.Objects.Widget#g:method:setFocusable"), [setFontMap]("GI.Gtk.Objects.Widget#g:method:setFontMap"), [setFontOptions]("GI.Gtk.Objects.Widget#g:method:setFontOptions"), [setGravity]("GI.Gtk.Objects.Window#g:method:setGravity"), [setHalign]("GI.Gtk.Objects.Widget#g:method:setHalign"), [setHandleMenubarAccel]("GI.Gtk.Objects.Window#g:method:setHandleMenubarAccel"), [setHasTooltip]("GI.Gtk.Objects.Widget#g:method:setHasTooltip"), [setHexpand]("GI.Gtk.Objects.Widget#g:method:setHexpand"), [setHexpandSet]("GI.Gtk.Objects.Widget#g:method:setHexpandSet"), [setHideOnClose]("GI.Gtk.Objects.Window#g:method:setHideOnClose"), [setIconName]("GI.Gtk.Objects.Window#g:method:setIconName"), [setLayoutManager]("GI.Gtk.Objects.Widget#g:method:setLayoutManager"), [setLimitEvents]("GI.Gtk.Objects.Widget#g:method:setLimitEvents"), [setMarginBottom]("GI.Gtk.Objects.Widget#g:method:setMarginBottom"), [setMarginEnd]("GI.Gtk.Objects.Widget#g:method:setMarginEnd"), [setMarginStart]("GI.Gtk.Objects.Widget#g:method:setMarginStart"), [setMarginTop]("GI.Gtk.Objects.Widget#g:method:setMarginTop"), [setMnemonicsVisible]("GI.Gtk.Objects.Window#g:method:setMnemonicsVisible"), [setModal]("GI.Gtk.Objects.Window#g:method:setModal"), [setName]("GI.Gtk.Objects.Widget#g:method:setName"), [setOpacity]("GI.Gtk.Objects.Widget#g:method:setOpacity"), [setOverflow]("GI.Gtk.Objects.Widget#g:method:setOverflow"), [setParent]("GI.Gtk.Objects.Widget#g:method:setParent"), [setProperty]("GI.GObject.Objects.Object#g:method:setProperty"), [setReceivesDefault]("GI.Gtk.Objects.Widget#g:method:setReceivesDefault"), [setResizable]("GI.Gtk.Objects.Window#g:method:setResizable"), [setResponseSensitive]("GI.Gtk.Objects.Dialog#g:method:setResponseSensitive"), [setSensitive]("GI.Gtk.Objects.Widget#g:method:setSensitive"), [setSizeRequest]("GI.Gtk.Objects.Widget#g:method:setSizeRequest"), [setStartupId]("GI.Gtk.Objects.Window#g:method:setStartupId"), [setStateFlags]("GI.Gtk.Objects.Widget#g:method:setStateFlags"), [setTitle]("GI.Gtk.Objects.Window#g:method:setTitle"), [setTitlebar]("GI.Gtk.Objects.Window#g:method:setTitlebar"), [setTooltipMarkup]("GI.Gtk.Objects.Widget#g:method:setTooltipMarkup"), [setTooltipText]("GI.Gtk.Objects.Widget#g:method:setTooltipText"), [setTransientFor]("GI.Gtk.Objects.Window#g:method:setTransientFor"), [setValign]("GI.Gtk.Objects.Widget#g:method:setValign"), [setVexpand]("GI.Gtk.Objects.Widget#g:method:setVexpand"), [setVexpandSet]("GI.Gtk.Objects.Widget#g:method:setVexpandSet"), [setVisible]("GI.Gtk.Objects.Widget#g:method:setVisible").

#if defined(ENABLE_OVERLOADING)
    ResolveDialogMethod                     ,
#endif

-- ** addActionWidget #method:addActionWidget#

#if defined(ENABLE_OVERLOADING)
    DialogAddActionWidgetMethodInfo         ,
#endif
    dialogAddActionWidget                   ,


-- ** addButton #method:addButton#

#if defined(ENABLE_OVERLOADING)
    DialogAddButtonMethodInfo               ,
#endif
    dialogAddButton                         ,


-- ** getContentArea #method:getContentArea#

#if defined(ENABLE_OVERLOADING)
    DialogGetContentAreaMethodInfo          ,
#endif
    dialogGetContentArea                    ,


-- ** getHeaderBar #method:getHeaderBar#

#if defined(ENABLE_OVERLOADING)
    DialogGetHeaderBarMethodInfo            ,
#endif
    dialogGetHeaderBar                      ,


-- ** getResponseForWidget #method:getResponseForWidget#

#if defined(ENABLE_OVERLOADING)
    DialogGetResponseForWidgetMethodInfo    ,
#endif
    dialogGetResponseForWidget              ,


-- ** getWidgetForResponse #method:getWidgetForResponse#

#if defined(ENABLE_OVERLOADING)
    DialogGetWidgetForResponseMethodInfo    ,
#endif
    dialogGetWidgetForResponse              ,


-- ** new #method:new#

    dialogNew                               ,


-- ** response #method:response#

#if defined(ENABLE_OVERLOADING)
    DialogResponseMethodInfo                ,
#endif
    dialogResponse                          ,


-- ** setDefaultResponse #method:setDefaultResponse#

#if defined(ENABLE_OVERLOADING)
    DialogSetDefaultResponseMethodInfo      ,
#endif
    dialogSetDefaultResponse                ,


-- ** setResponseSensitive #method:setResponseSensitive#

#if defined(ENABLE_OVERLOADING)
    DialogSetResponseSensitiveMethodInfo    ,
#endif
    dialogSetResponseSensitive              ,




 -- * Properties


-- ** useHeaderBar #attr:useHeaderBar#
-- | 'P.True' if the dialog uses a headerbar for action buttons
-- instead of the action-area.
-- 
-- For technical reasons, this property is declared as an integer
-- property, but you should only set it to 'P.True' or 'P.False'.
-- 
-- == Creating a dialog with headerbar
-- 
-- Builtin @GtkDialog@ subclasses such as t'GI.Gtk.Objects.ColorChooserDialog.ColorChooserDialog'
-- set this property according to platform conventions (using the
-- [Settings:gtkDialogsUseHeader]("GI.Gtk.Objects.Settings#g:attr:gtkDialogsUseHeader") setting).
-- 
-- Here is how you can achieve the same:
-- 
-- 
-- === /c code/
-- >g_object_get (settings, "gtk-dialogs-use-header", &header, NULL);
-- >dialog = g_object_new (GTK_TYPE_DIALOG, header, TRUE, NULL);

#if defined(ENABLE_OVERLOADING)
    DialogUseHeaderBarPropertyInfo          ,
#endif
    constructDialogUseHeaderBar             ,
#if defined(ENABLE_OVERLOADING)
    dialogUseHeaderBar                      ,
#endif
    getDialogUseHeaderBar                   ,




 -- * Signals


-- ** close #signal:close#

    DialogCloseCallback                     ,
#if defined(ENABLE_OVERLOADING)
    DialogCloseSignalInfo                   ,
#endif
    afterDialogClose                        ,
    onDialogClose                           ,


-- ** response #signal:response#

    DialogResponseCallback                  ,
#if defined(ENABLE_OVERLOADING)
    DialogResponseSignalInfo                ,
#endif
    afterDialogResponse                     ,
    onDialogResponse                        ,




    ) where

import Data.GI.Base.ShortPrelude
import qualified Data.GI.Base.ShortPrelude as SP
import qualified Data.GI.Base.Overloading as O
import qualified Prelude as P

import qualified Data.GI.Base.Attributes as GI.Attributes
import qualified Data.GI.Base.BasicTypes as B.Types
import qualified Data.GI.Base.ManagedPtr as B.ManagedPtr
import qualified Data.GI.Base.GArray as B.GArray
import qualified Data.GI.Base.GClosure as B.GClosure
import qualified Data.GI.Base.GError as B.GError
import qualified Data.GI.Base.GHashTable as B.GHT
import qualified Data.GI.Base.GVariant as B.GVariant
import qualified Data.GI.Base.GValue as B.GValue
import qualified Data.GI.Base.GParamSpec as B.GParamSpec
import qualified Data.GI.Base.CallStack as B.CallStack
import qualified Data.GI.Base.Properties as B.Properties
import qualified Data.GI.Base.Signals as B.Signals
import qualified Control.Monad.IO.Class as MIO
import qualified Data.Coerce as Coerce
import qualified Data.Text as T
import qualified Data.Kind as DK
import qualified Data.ByteString.Char8 as B
import qualified Data.Map as Map
import qualified Foreign.Ptr as FP
import qualified GHC.OverloadedLabels as OL
import qualified GHC.Records as R
import qualified Data.Word as DW
import qualified Data.Int as DI
import qualified System.Posix.Types as SPT
import qualified Foreign.C.Types as FCT

-- Workaround for https://gitlab.haskell.org/ghc/ghc/-/issues/23392
#if MIN_VERSION_base(4,18,0)
import qualified GI.Cairo.Structs.Context as Cairo.Context
import qualified GI.Cairo.Structs.FontOptions as Cairo.FontOptions
import qualified GI.GLib.Callbacks as GLib.Callbacks
import qualified GI.GLib.Structs.Bytes as GLib.Bytes
import qualified GI.GObject.Objects.Object as GObject.Object
import qualified GI.Gdk.Flags as Gdk.Flags
import qualified GI.Gdk.Interfaces.Paintable as Gdk.Paintable
import qualified GI.Gdk.Objects.Clipboard as Gdk.Clipboard
import qualified GI.Gdk.Objects.Cursor as Gdk.Cursor
import qualified GI.Gdk.Objects.Device as Gdk.Device
import qualified GI.Gdk.Objects.Display as Gdk.Display
import qualified GI.Gdk.Objects.Event as Gdk.Event
import qualified GI.Gdk.Objects.FrameClock as Gdk.FrameClock
import qualified GI.Gdk.Objects.Monitor as Gdk.Monitor
import qualified GI.Gdk.Objects.Snapshot as Gdk.Snapshot
import qualified GI.Gdk.Objects.Surface as Gdk.Surface
import qualified GI.Gdk.Objects.Texture as Gdk.Texture
import qualified GI.Gdk.Structs.RGBA as Gdk.RGBA
import qualified GI.Gdk.Structs.Rectangle as Gdk.Rectangle
import qualified GI.Gio.Flags as Gio.Flags
import qualified GI.Gio.Interfaces.ActionGroup as Gio.ActionGroup
import qualified GI.Gio.Interfaces.ActionMap as Gio.ActionMap
import qualified GI.Gio.Interfaces.Icon as Gio.Icon
import qualified GI.Gio.Interfaces.ListModel as Gio.ListModel
import qualified GI.Gio.Objects.Application as Gio.Application
import qualified GI.Gio.Objects.Menu as Gio.Menu
import qualified GI.Gio.Objects.MenuModel as Gio.MenuModel
import qualified GI.Graphene.Structs.Matrix as Graphene.Matrix
import qualified GI.Graphene.Structs.Point as Graphene.Point
import qualified GI.Graphene.Structs.Point3D as Graphene.Point3D
import qualified GI.Graphene.Structs.Rect as Graphene.Rect
import qualified GI.Graphene.Structs.Size as Graphene.Size
import qualified GI.Graphene.Structs.Vec3 as Graphene.Vec3
import qualified GI.Graphene.Structs.Vec4 as Graphene.Vec4
import qualified GI.Gsk.Enums as Gsk.Enums
import qualified GI.Gsk.Objects.GLShader as Gsk.GLShader
import qualified GI.Gsk.Objects.RenderNode as Gsk.RenderNode
import qualified GI.Gsk.Objects.Renderer as Gsk.Renderer
import qualified GI.Gsk.Structs.ColorStop as Gsk.ColorStop
import qualified GI.Gsk.Structs.ComponentTransfer as Gsk.ComponentTransfer
import qualified GI.Gsk.Structs.Path as Gsk.Path
import qualified GI.Gsk.Structs.RoundedRect as Gsk.RoundedRect
import qualified GI.Gsk.Structs.Shadow as Gsk.Shadow
import qualified GI.Gsk.Structs.Stroke as Gsk.Stroke
import qualified GI.Gsk.Structs.Transform as Gsk.Transform
import qualified GI.Gtk.Callbacks as Gtk.Callbacks
import {-# SOURCE #-} qualified GI.Gtk.Enums as Gtk.Enums
import {-# SOURCE #-} qualified GI.Gtk.Flags as Gtk.Flags
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Accessible as Gtk.Accessible
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.ConstraintTarget as Gtk.ConstraintTarget
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Native as Gtk.Native
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Orientable as Gtk.Orientable
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Root as Gtk.Root
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.ShortcutManager as Gtk.ShortcutManager
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.StyleProvider as Gtk.StyleProvider
import {-# SOURCE #-} qualified GI.Gtk.Objects.ATContext as Gtk.ATContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.Application as Gtk.Application
import {-# SOURCE #-} qualified GI.Gtk.Objects.Box as Gtk.Box
import {-# SOURCE #-} qualified GI.Gtk.Objects.EventController as Gtk.EventController
import {-# SOURCE #-} qualified GI.Gtk.Objects.HeaderBar as Gtk.HeaderBar
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutChild as Gtk.LayoutChild
import {-# SOURCE #-} qualified GI.Gtk.Objects.LayoutManager as Gtk.LayoutManager
import {-# SOURCE #-} qualified GI.Gtk.Objects.Settings as Gtk.Settings
import {-# SOURCE #-} qualified GI.Gtk.Objects.Snapshot as Gtk.Snapshot
import {-# SOURCE #-} qualified GI.Gtk.Objects.StyleContext as Gtk.StyleContext
import {-# SOURCE #-} qualified GI.Gtk.Objects.Tooltip as Gtk.Tooltip
import {-# SOURCE #-} qualified GI.Gtk.Objects.Widget as Gtk.Widget
import {-# SOURCE #-} qualified GI.Gtk.Objects.Window as Gtk.Window
import {-# SOURCE #-} qualified GI.Gtk.Objects.WindowGroup as Gtk.WindowGroup
import {-# SOURCE #-} qualified GI.Gtk.Structs.Border as Gtk.Border
import {-# SOURCE #-} qualified GI.Gtk.Structs.Requisition as Gtk.Requisition
import qualified GI.Pango.Enums as Pango.Enums
import qualified GI.Pango.Objects.Context as Pango.Context
import qualified GI.Pango.Objects.FontMap as Pango.FontMap
import qualified GI.Pango.Objects.Layout as Pango.Layout

#else
import qualified GI.GObject.Objects.Object as GObject.Object
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Accessible as Gtk.Accessible
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Buildable as Gtk.Buildable
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.ConstraintTarget as Gtk.ConstraintTarget
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Native as Gtk.Native
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.Root as Gtk.Root
import {-# SOURCE #-} qualified GI.Gtk.Interfaces.ShortcutManager as Gtk.ShortcutManager
import {-# SOURCE #-} qualified GI.Gtk.Objects.Box as Gtk.Box
import {-# SOURCE #-} qualified GI.Gtk.Objects.HeaderBar as Gtk.HeaderBar
import {-# SOURCE #-} qualified GI.Gtk.Objects.Widget as Gtk.Widget
import {-# SOURCE #-} qualified GI.Gtk.Objects.Window as Gtk.Window

#endif

-- | Memory-managed wrapper type.
newtype Dialog = Dialog (SP.ManagedPtr Dialog)
    deriving (Eq)

instance SP.ManagedPtrNewtype Dialog where
    toManagedPtr (Dialog p) = p

foreign import ccall "gtk_dialog_get_type"
    c_gtk_dialog_get_type :: IO B.Types.GType

instance B.Types.TypedObject Dialog where
    glibType = c_gtk_dialog_get_type

instance B.Types.GObject Dialog

-- | Type class for types which can be safely cast to t'Dialog', for instance with `toDialog`.
class (SP.GObject o, O.IsDescendantOf Dialog o) => IsDialog o
instance (SP.GObject o, O.IsDescendantOf Dialog o) => IsDialog o

instance O.HasParentTypes Dialog
type instance O.ParentTypes Dialog = '[Gtk.Window.Window, Gtk.Widget.Widget, GObject.Object.Object, Gtk.Accessible.Accessible, Gtk.Buildable.Buildable, Gtk.ConstraintTarget.ConstraintTarget, Gtk.Native.Native, Gtk.Root.Root, Gtk.ShortcutManager.ShortcutManager]

-- | Cast to t'Dialog', for types for which this is known to be safe. For general casts, use 'Data.GI.Base.ManagedPtr.castTo'.
toDialog :: (MIO.MonadIO m, IsDialog o) => o -> m Dialog
toDialog = MIO.liftIO . B.ManagedPtr.unsafeCastTo Dialog

-- | Convert t'Dialog' to and from t'Data.GI.Base.GValue.GValue'. See 'Data.GI.Base.GValue.toGValue' and 'Data.GI.Base.GValue.fromGValue'.
instance B.GValue.IsGValue (Maybe Dialog) where
    gvalueGType_ = c_gtk_dialog_get_type
    gvalueSet_ gv P.Nothing = B.GValue.set_object gv (FP.nullPtr :: FP.Ptr Dialog)
    gvalueSet_ gv (P.Just obj) = B.ManagedPtr.withManagedPtr obj (B.GValue.set_object gv)
    gvalueGet_ gv = do
        ptr <- B.GValue.get_object gv :: IO (FP.Ptr Dialog)
        if ptr /= FP.nullPtr
        then P.Just <$> B.ManagedPtr.newObject Dialog ptr
        else return P.Nothing
        
    

#if defined(ENABLE_OVERLOADING)
type family ResolveDialogMethod (t :: Symbol) (o :: DK.Type) :: DK.Type where
    ResolveDialogMethod "actionSetEnabled" o = Gtk.Widget.WidgetActionSetEnabledMethodInfo
    ResolveDialogMethod "activate" o = Gtk.Widget.WidgetActivateMethodInfo
    ResolveDialogMethod "activateAction" o = Gtk.Widget.WidgetActivateActionMethodInfo
    ResolveDialogMethod "activateDefault" o = Gtk.Widget.WidgetActivateDefaultMethodInfo
    ResolveDialogMethod "addActionWidget" o = DialogAddActionWidgetMethodInfo
    ResolveDialogMethod "addButton" o = DialogAddButtonMethodInfo
    ResolveDialogMethod "addController" o = Gtk.Widget.WidgetAddControllerMethodInfo
    ResolveDialogMethod "addCssClass" o = Gtk.Widget.WidgetAddCssClassMethodInfo
    ResolveDialogMethod "addMnemonicLabel" o = Gtk.Widget.WidgetAddMnemonicLabelMethodInfo
    ResolveDialogMethod "addTickCallback" o = Gtk.Widget.WidgetAddTickCallbackMethodInfo
    ResolveDialogMethod "allocate" o = Gtk.Widget.WidgetAllocateMethodInfo
    ResolveDialogMethod "announce" o = Gtk.Accessible.AccessibleAnnounceMethodInfo
    ResolveDialogMethod "bindProperty" o = GObject.Object.ObjectBindPropertyMethodInfo
    ResolveDialogMethod "bindPropertyFull" o = GObject.Object.ObjectBindPropertyFullMethodInfo
    ResolveDialogMethod "childFocus" o = Gtk.Widget.WidgetChildFocusMethodInfo
    ResolveDialogMethod "close" o = Gtk.Window.WindowCloseMethodInfo
    ResolveDialogMethod "computeBounds" o = Gtk.Widget.WidgetComputeBoundsMethodInfo
    ResolveDialogMethod "computeExpand" o = Gtk.Widget.WidgetComputeExpandMethodInfo
    ResolveDialogMethod "computePoint" o = Gtk.Widget.WidgetComputePointMethodInfo
    ResolveDialogMethod "computeTransform" o = Gtk.Widget.WidgetComputeTransformMethodInfo
    ResolveDialogMethod "contains" o = Gtk.Widget.WidgetContainsMethodInfo
    ResolveDialogMethod "createPangoContext" o = Gtk.Widget.WidgetCreatePangoContextMethodInfo
    ResolveDialogMethod "createPangoLayout" o = Gtk.Widget.WidgetCreatePangoLayoutMethodInfo
    ResolveDialogMethod "destroy" o = Gtk.Window.WindowDestroyMethodInfo
    ResolveDialogMethod "disposeTemplate" o = Gtk.Widget.WidgetDisposeTemplateMethodInfo
    ResolveDialogMethod "dragCheckThreshold" o = Gtk.Widget.WidgetDragCheckThresholdMethodInfo
    ResolveDialogMethod "errorBell" o = Gtk.Widget.WidgetErrorBellMethodInfo
    ResolveDialogMethod "forceFloating" o = GObject.Object.ObjectForceFloatingMethodInfo
    ResolveDialogMethod "freezeNotify" o = GObject.Object.ObjectFreezeNotifyMethodInfo
    ResolveDialogMethod "fullscreen" o = Gtk.Window.WindowFullscreenMethodInfo
    ResolveDialogMethod "fullscreenOnMonitor" o = Gtk.Window.WindowFullscreenOnMonitorMethodInfo
    ResolveDialogMethod "getv" o = GObject.Object.ObjectGetvMethodInfo
    ResolveDialogMethod "grabFocus" o = Gtk.Widget.WidgetGrabFocusMethodInfo
    ResolveDialogMethod "hasCssClass" o = Gtk.Widget.WidgetHasCssClassMethodInfo
    ResolveDialogMethod "hasDefault" o = Gtk.Widget.WidgetHasDefaultMethodInfo
    ResolveDialogMethod "hasFocus" o = Gtk.Widget.WidgetHasFocusMethodInfo
    ResolveDialogMethod "hasGroup" o = Gtk.Window.WindowHasGroupMethodInfo
    ResolveDialogMethod "hasVisibleFocus" o = Gtk.Widget.WidgetHasVisibleFocusMethodInfo
    ResolveDialogMethod "hide" o = Gtk.Widget.WidgetHideMethodInfo
    ResolveDialogMethod "inDestruction" o = Gtk.Widget.WidgetInDestructionMethodInfo
    ResolveDialogMethod "initTemplate" o = Gtk.Widget.WidgetInitTemplateMethodInfo
    ResolveDialogMethod "insertActionGroup" o = Gtk.Widget.WidgetInsertActionGroupMethodInfo
    ResolveDialogMethod "insertAfter" o = Gtk.Widget.WidgetInsertAfterMethodInfo
    ResolveDialogMethod "insertBefore" o = Gtk.Widget.WidgetInsertBeforeMethodInfo
    ResolveDialogMethod "isActive" o = Gtk.Window.WindowIsActiveMethodInfo
    ResolveDialogMethod "isAncestor" o = Gtk.Widget.WidgetIsAncestorMethodInfo
    ResolveDialogMethod "isDrawable" o = Gtk.Widget.WidgetIsDrawableMethodInfo
    ResolveDialogMethod "isFloating" o = GObject.Object.ObjectIsFloatingMethodInfo
    ResolveDialogMethod "isFocus" o = Gtk.Widget.WidgetIsFocusMethodInfo
    ResolveDialogMethod "isFullscreen" o = Gtk.Window.WindowIsFullscreenMethodInfo
    ResolveDialogMethod "isMaximized" o = Gtk.Window.WindowIsMaximizedMethodInfo
    ResolveDialogMethod "isSensitive" o = Gtk.Widget.WidgetIsSensitiveMethodInfo
    ResolveDialogMethod "isSuspended" o = Gtk.Window.WindowIsSuspendedMethodInfo
    ResolveDialogMethod "isVisible" o = Gtk.Widget.WidgetIsVisibleMethodInfo
    ResolveDialogMethod "keynavFailed" o = Gtk.Widget.WidgetKeynavFailedMethodInfo
    ResolveDialogMethod "listMnemonicLabels" o = Gtk.Widget.WidgetListMnemonicLabelsMethodInfo
    ResolveDialogMethod "map" o = Gtk.Widget.WidgetMapMethodInfo
    ResolveDialogMethod "maximize" o = Gtk.Window.WindowMaximizeMethodInfo
    ResolveDialogMethod "measure" o = Gtk.Widget.WidgetMeasureMethodInfo
    ResolveDialogMethod "minimize" o = Gtk.Window.WindowMinimizeMethodInfo
    ResolveDialogMethod "mnemonicActivate" o = Gtk.Widget.WidgetMnemonicActivateMethodInfo
    ResolveDialogMethod "notify" o = GObject.Object.ObjectNotifyMethodInfo
    ResolveDialogMethod "notifyByPspec" o = GObject.Object.ObjectNotifyByPspecMethodInfo
    ResolveDialogMethod "observeChildren" o = Gtk.Widget.WidgetObserveChildrenMethodInfo
    ResolveDialogMethod "observeControllers" o = Gtk.Widget.WidgetObserveControllersMethodInfo
    ResolveDialogMethod "pick" o = Gtk.Widget.WidgetPickMethodInfo
    ResolveDialogMethod "present" o = Gtk.Window.WindowPresentMethodInfo
    ResolveDialogMethod "presentWithTime" o = Gtk.Window.WindowPresentWithTimeMethodInfo
    ResolveDialogMethod "queueAllocate" o = Gtk.Widget.WidgetQueueAllocateMethodInfo
    ResolveDialogMethod "queueDraw" o = Gtk.Widget.WidgetQueueDrawMethodInfo
    ResolveDialogMethod "queueResize" o = Gtk.Widget.WidgetQueueResizeMethodInfo
    ResolveDialogMethod "realize" o = Gtk.Widget.WidgetRealizeMethodInfo
    ResolveDialogMethod "ref" o = GObject.Object.ObjectRefMethodInfo
    ResolveDialogMethod "refSink" o = GObject.Object.ObjectRefSinkMethodInfo
    ResolveDialogMethod "removeController" o = Gtk.Widget.WidgetRemoveControllerMethodInfo
    ResolveDialogMethod "removeCssClass" o = Gtk.Widget.WidgetRemoveCssClassMethodInfo
    ResolveDialogMethod "removeMnemonicLabel" o = Gtk.Widget.WidgetRemoveMnemonicLabelMethodInfo
    ResolveDialogMethod "removeTickCallback" o = Gtk.Widget.WidgetRemoveTickCallbackMethodInfo
    ResolveDialogMethod "resetProperty" o = Gtk.Accessible.AccessibleResetPropertyMethodInfo
    ResolveDialogMethod "resetRelation" o = Gtk.Accessible.AccessibleResetRelationMethodInfo
    ResolveDialogMethod "resetState" o = Gtk.Accessible.AccessibleResetStateMethodInfo
    ResolveDialogMethod "response" o = DialogResponseMethodInfo
    ResolveDialogMethod "runDispose" o = GObject.Object.ObjectRunDisposeMethodInfo
    ResolveDialogMethod "shouldLayout" o = Gtk.Widget.WidgetShouldLayoutMethodInfo
    ResolveDialogMethod "show" o = Gtk.Widget.WidgetShowMethodInfo
    ResolveDialogMethod "sizeAllocate" o = Gtk.Widget.WidgetSizeAllocateMethodInfo
    ResolveDialogMethod "snapshotChild" o = Gtk.Widget.WidgetSnapshotChildMethodInfo
    ResolveDialogMethod "stealData" o = GObject.Object.ObjectStealDataMethodInfo
    ResolveDialogMethod "stealQdata" o = GObject.Object.ObjectStealQdataMethodInfo
    ResolveDialogMethod "thawNotify" o = GObject.Object.ObjectThawNotifyMethodInfo
    ResolveDialogMethod "translateCoordinates" o = Gtk.Widget.WidgetTranslateCoordinatesMethodInfo
    ResolveDialogMethod "triggerTooltipQuery" o = Gtk.Widget.WidgetTriggerTooltipQueryMethodInfo
    ResolveDialogMethod "unfullscreen" o = Gtk.Window.WindowUnfullscreenMethodInfo
    ResolveDialogMethod "unmap" o = Gtk.Widget.WidgetUnmapMethodInfo
    ResolveDialogMethod "unmaximize" o = Gtk.Window.WindowUnmaximizeMethodInfo
    ResolveDialogMethod "unminimize" o = Gtk.Window.WindowUnminimizeMethodInfo
    ResolveDialogMethod "unparent" o = Gtk.Widget.WidgetUnparentMethodInfo
    ResolveDialogMethod "unrealize" o = Gtk.Widget.WidgetUnrealizeMethodInfo
    ResolveDialogMethod "unref" o = GObject.Object.ObjectUnrefMethodInfo
    ResolveDialogMethod "unsetStateFlags" o = Gtk.Widget.WidgetUnsetStateFlagsMethodInfo
    ResolveDialogMethod "updateNextAccessibleSibling" o = Gtk.Accessible.AccessibleUpdateNextAccessibleSiblingMethodInfo
    ResolveDialogMethod "updatePlatformState" o = Gtk.Accessible.AccessibleUpdatePlatformStateMethodInfo
    ResolveDialogMethod "updateProperty" o = Gtk.Accessible.AccessibleUpdatePropertyMethodInfo
    ResolveDialogMethod "updateRelation" o = Gtk.Accessible.AccessibleUpdateRelationMethodInfo
    ResolveDialogMethod "updateState" o = Gtk.Accessible.AccessibleUpdateStateMethodInfo
    ResolveDialogMethod "watchClosure" o = GObject.Object.ObjectWatchClosureMethodInfo
    ResolveDialogMethod "getAccessibleParent" o = Gtk.Accessible.AccessibleGetAccessibleParentMethodInfo
    ResolveDialogMethod "getAccessibleRole" o = Gtk.Accessible.AccessibleGetAccessibleRoleMethodInfo
    ResolveDialogMethod "getAllocatedBaseline" o = Gtk.Widget.WidgetGetAllocatedBaselineMethodInfo
    ResolveDialogMethod "getAllocatedHeight" o = Gtk.Widget.WidgetGetAllocatedHeightMethodInfo
    ResolveDialogMethod "getAllocatedWidth" o = Gtk.Widget.WidgetGetAllocatedWidthMethodInfo
    ResolveDialogMethod "getAllocation" o = Gtk.Widget.WidgetGetAllocationMethodInfo
    ResolveDialogMethod "getAncestor" o = Gtk.Widget.WidgetGetAncestorMethodInfo
    ResolveDialogMethod "getApplication" o = Gtk.Window.WindowGetApplicationMethodInfo
    ResolveDialogMethod "getAtContext" o = Gtk.Accessible.AccessibleGetAtContextMethodInfo
    ResolveDialogMethod "getBaseline" o = Gtk.Widget.WidgetGetBaselineMethodInfo
    ResolveDialogMethod "getBounds" o = Gtk.Accessible.AccessibleGetBoundsMethodInfo
    ResolveDialogMethod "getBuildableId" o = Gtk.Buildable.BuildableGetBuildableIdMethodInfo
    ResolveDialogMethod "getCanFocus" o = Gtk.Widget.WidgetGetCanFocusMethodInfo
    ResolveDialogMethod "getCanTarget" o = Gtk.Widget.WidgetGetCanTargetMethodInfo
    ResolveDialogMethod "getChild" o = Gtk.Window.WindowGetChildMethodInfo
    ResolveDialogMethod "getChildVisible" o = Gtk.Widget.WidgetGetChildVisibleMethodInfo
    ResolveDialogMethod "getClipboard" o = Gtk.Widget.WidgetGetClipboardMethodInfo
    ResolveDialogMethod "getColor" o = Gtk.Widget.WidgetGetColorMethodInfo
    ResolveDialogMethod "getContentArea" o = DialogGetContentAreaMethodInfo
    ResolveDialogMethod "getCssClasses" o = Gtk.Widget.WidgetGetCssClassesMethodInfo
    ResolveDialogMethod "getCssName" o = Gtk.Widget.WidgetGetCssNameMethodInfo
    ResolveDialogMethod "getCursor" o = Gtk.Widget.WidgetGetCursorMethodInfo
    ResolveDialogMethod "getData" o = GObject.Object.ObjectGetDataMethodInfo
    ResolveDialogMethod "getDecorated" o = Gtk.Window.WindowGetDecoratedMethodInfo
    ResolveDialogMethod "getDefaultSize" o = Gtk.Window.WindowGetDefaultSizeMethodInfo
    ResolveDialogMethod "getDefaultWidget" o = Gtk.Window.WindowGetDefaultWidgetMethodInfo
    ResolveDialogMethod "getDeletable" o = Gtk.Window.WindowGetDeletableMethodInfo
    ResolveDialogMethod "getDestroyWithParent" o = Gtk.Window.WindowGetDestroyWithParentMethodInfo
    ResolveDialogMethod "getDirection" o = Gtk.Widget.WidgetGetDirectionMethodInfo
    ResolveDialogMethod "getDisplay" o = Gtk.Widget.WidgetGetDisplayMethodInfo
    ResolveDialogMethod "getFirstAccessibleChild" o = Gtk.Accessible.AccessibleGetFirstAccessibleChildMethodInfo
    ResolveDialogMethod "getFirstChild" o = Gtk.Widget.WidgetGetFirstChildMethodInfo
    ResolveDialogMethod "getFocus" o = Gtk.Window.WindowGetFocusMethodInfo
    ResolveDialogMethod "getFocusChild" o = Gtk.Widget.WidgetGetFocusChildMethodInfo
    ResolveDialogMethod "getFocusOnClick" o = Gtk.Widget.WidgetGetFocusOnClickMethodInfo
    ResolveDialogMethod "getFocusVisible" o = Gtk.Window.WindowGetFocusVisibleMethodInfo
    ResolveDialogMethod "getFocusable" o = Gtk.Widget.WidgetGetFocusableMethodInfo
    ResolveDialogMethod "getFontMap" o = Gtk.Widget.WidgetGetFontMapMethodInfo
    ResolveDialogMethod "getFontOptions" o = Gtk.Widget.WidgetGetFontOptionsMethodInfo
    ResolveDialogMethod "getFrameClock" o = Gtk.Widget.WidgetGetFrameClockMethodInfo
    ResolveDialogMethod "getGravity" o = Gtk.Window.WindowGetGravityMethodInfo
    ResolveDialogMethod "getGroup" o = Gtk.Window.WindowGetGroupMethodInfo
    ResolveDialogMethod "getHalign" o = Gtk.Widget.WidgetGetHalignMethodInfo
    ResolveDialogMethod "getHandleMenubarAccel" o = Gtk.Window.WindowGetHandleMenubarAccelMethodInfo
    ResolveDialogMethod "getHasTooltip" o = Gtk.Widget.WidgetGetHasTooltipMethodInfo
    ResolveDialogMethod "getHeaderBar" o = DialogGetHeaderBarMethodInfo
    ResolveDialogMethod "getHeight" o = Gtk.Widget.WidgetGetHeightMethodInfo
    ResolveDialogMethod "getHexpand" o = Gtk.Widget.WidgetGetHexpandMethodInfo
    ResolveDialogMethod "getHexpandSet" o = Gtk.Widget.WidgetGetHexpandSetMethodInfo
    ResolveDialogMethod "getHideOnClose" o = Gtk.Window.WindowGetHideOnCloseMethodInfo
    ResolveDialogMethod "getIconName" o = Gtk.Window.WindowGetIconNameMethodInfo
    ResolveDialogMethod "getLastChild" o = Gtk.Widget.WidgetGetLastChildMethodInfo
    ResolveDialogMethod "getLayoutManager" o = Gtk.Widget.WidgetGetLayoutManagerMethodInfo
    ResolveDialogMethod "getLimitEvents" o = Gtk.Widget.WidgetGetLimitEventsMethodInfo
    ResolveDialogMethod "getMapped" o = Gtk.Widget.WidgetGetMappedMethodInfo
    ResolveDialogMethod "getMarginBottom" o = Gtk.Widget.WidgetGetMarginBottomMethodInfo
    ResolveDialogMethod "getMarginEnd" o = Gtk.Widget.WidgetGetMarginEndMethodInfo
    ResolveDialogMethod "getMarginStart" o = Gtk.Widget.WidgetGetMarginStartMethodInfo
    ResolveDialogMethod "getMarginTop" o = Gtk.Widget.WidgetGetMarginTopMethodInfo
    ResolveDialogMethod "getMnemonicsVisible" o = Gtk.Window.WindowGetMnemonicsVisibleMethodInfo
    ResolveDialogMethod "getModal" o = Gtk.Window.WindowGetModalMethodInfo
    ResolveDialogMethod "getName" o = Gtk.Widget.WidgetGetNameMethodInfo
    ResolveDialogMethod "getNative" o = Gtk.Widget.WidgetGetNativeMethodInfo
    ResolveDialogMethod "getNextAccessibleSibling" o = Gtk.Accessible.AccessibleGetNextAccessibleSiblingMethodInfo
    ResolveDialogMethod "getNextSibling" o = Gtk.Widget.WidgetGetNextSiblingMethodInfo
    ResolveDialogMethod "getOpacity" o = Gtk.Widget.WidgetGetOpacityMethodInfo
    ResolveDialogMethod "getOverflow" o = Gtk.Widget.WidgetGetOverflowMethodInfo
    ResolveDialogMethod "getPangoContext" o = Gtk.Widget.WidgetGetPangoContextMethodInfo
    ResolveDialogMethod "getParent" o = Gtk.Widget.WidgetGetParentMethodInfo
    ResolveDialogMethod "getPlatformState" o = Gtk.Accessible.AccessibleGetPlatformStateMethodInfo
    ResolveDialogMethod "getPreferredSize" o = Gtk.Widget.WidgetGetPreferredSizeMethodInfo
    ResolveDialogMethod "getPrevSibling" o = Gtk.Widget.WidgetGetPrevSiblingMethodInfo
    ResolveDialogMethod "getPrimaryClipboard" o = Gtk.Widget.WidgetGetPrimaryClipboardMethodInfo
    ResolveDialogMethod "getProperty" o = GObject.Object.ObjectGetPropertyMethodInfo
    ResolveDialogMethod "getQdata" o = GObject.Object.ObjectGetQdataMethodInfo
    ResolveDialogMethod "getRealized" o = Gtk.Widget.WidgetGetRealizedMethodInfo
    ResolveDialogMethod "getReceivesDefault" o = Gtk.Widget.WidgetGetReceivesDefaultMethodInfo
    ResolveDialogMethod "getRenderer" o = Gtk.Native.NativeGetRendererMethodInfo
    ResolveDialogMethod "getRequestMode" o = Gtk.Widget.WidgetGetRequestModeMethodInfo
    ResolveDialogMethod "getResizable" o = Gtk.Window.WindowGetResizableMethodInfo
    ResolveDialogMethod "getResponseForWidget" o = DialogGetResponseForWidgetMethodInfo
    ResolveDialogMethod "getRoot" o = Gtk.Widget.WidgetGetRootMethodInfo
    ResolveDialogMethod "getScaleFactor" o = Gtk.Widget.WidgetGetScaleFactorMethodInfo
    ResolveDialogMethod "getSensitive" o = Gtk.Widget.WidgetGetSensitiveMethodInfo
    ResolveDialogMethod "getSettings" o = Gtk.Widget.WidgetGetSettingsMethodInfo
    ResolveDialogMethod "getSize" o = Gtk.Widget.WidgetGetSizeMethodInfo
    ResolveDialogMethod "getSizeRequest" o = Gtk.Widget.WidgetGetSizeRequestMethodInfo
    ResolveDialogMethod "getStateFlags" o = Gtk.Widget.WidgetGetStateFlagsMethodInfo
    ResolveDialogMethod "getStyleContext" o = Gtk.Widget.WidgetGetStyleContextMethodInfo
    ResolveDialogMethod "getSurface" o = Gtk.Native.NativeGetSurfaceMethodInfo
    ResolveDialogMethod "getSurfaceTransform" o = Gtk.Native.NativeGetSurfaceTransformMethodInfo
    ResolveDialogMethod "getTemplateChild" o = Gtk.Widget.WidgetGetTemplateChildMethodInfo
    ResolveDialogMethod "getTitle" o = Gtk.Window.WindowGetTitleMethodInfo
    ResolveDialogMethod "getTitlebar" o = Gtk.Window.WindowGetTitlebarMethodInfo
    ResolveDialogMethod "getTooltipMarkup" o = Gtk.Widget.WidgetGetTooltipMarkupMethodInfo
    ResolveDialogMethod "getTooltipText" o = Gtk.Widget.WidgetGetTooltipTextMethodInfo
    ResolveDialogMethod "getTransientFor" o = Gtk.Window.WindowGetTransientForMethodInfo
    ResolveDialogMethod "getValign" o = Gtk.Widget.WidgetGetValignMethodInfo
    ResolveDialogMethod "getVexpand" o = Gtk.Widget.WidgetGetVexpandMethodInfo
    ResolveDialogMethod "getVexpandSet" o = Gtk.Widget.WidgetGetVexpandSetMethodInfo
    ResolveDialogMethod "getVisible" o = Gtk.Widget.WidgetGetVisibleMethodInfo
    ResolveDialogMethod "getWidgetForResponse" o = DialogGetWidgetForResponseMethodInfo
    ResolveDialogMethod "getWidth" o = Gtk.Widget.WidgetGetWidthMethodInfo
    ResolveDialogMethod "setAccessibleParent" o = Gtk.Accessible.AccessibleSetAccessibleParentMethodInfo
    ResolveDialogMethod "setApplication" o = Gtk.Window.WindowSetApplicationMethodInfo
    ResolveDialogMethod "setCanFocus" o = Gtk.Widget.WidgetSetCanFocusMethodInfo
    ResolveDialogMethod "setCanTarget" o = Gtk.Widget.WidgetSetCanTargetMethodInfo
    ResolveDialogMethod "setChild" o = Gtk.Window.WindowSetChildMethodInfo
    ResolveDialogMethod "setChildVisible" o = Gtk.Widget.WidgetSetChildVisibleMethodInfo
    ResolveDialogMethod "setCssClasses" o = Gtk.Widget.WidgetSetCssClassesMethodInfo
    ResolveDialogMethod "setCursor" o = Gtk.Widget.WidgetSetCursorMethodInfo
    ResolveDialogMethod "setCursorFromName" o = Gtk.Widget.WidgetSetCursorFromNameMethodInfo
    ResolveDialogMethod "setData" o = GObject.Object.ObjectSetDataMethodInfo
    ResolveDialogMethod "setDataFull" o = GObject.Object.ObjectSetDataFullMethodInfo
    ResolveDialogMethod "setDecorated" o = Gtk.Window.WindowSetDecoratedMethodInfo
    ResolveDialogMethod "setDefaultResponse" o = DialogSetDefaultResponseMethodInfo
    ResolveDialogMethod "setDefaultSize" o = Gtk.Window.WindowSetDefaultSizeMethodInfo
    ResolveDialogMethod "setDefaultWidget" o = Gtk.Window.WindowSetDefaultWidgetMethodInfo
    ResolveDialogMethod "setDeletable" o = Gtk.Window.WindowSetDeletableMethodInfo
    ResolveDialogMethod "setDestroyWithParent" o = Gtk.Window.WindowSetDestroyWithParentMethodInfo
    ResolveDialogMethod "setDirection" o = Gtk.Widget.WidgetSetDirectionMethodInfo
    ResolveDialogMethod "setDisplay" o = Gtk.Window.WindowSetDisplayMethodInfo
    ResolveDialogMethod "setFocus" o = Gtk.Window.WindowSetFocusMethodInfo
    ResolveDialogMethod "setFocusChild" o = Gtk.Widget.WidgetSetFocusChildMethodInfo
    ResolveDialogMethod "setFocusOnClick" o = Gtk.Widget.WidgetSetFocusOnClickMethodInfo
    ResolveDialogMethod "setFocusVisible" o = Gtk.Window.WindowSetFocusVisibleMethodInfo
    ResolveDialogMethod "setFocusable" o = Gtk.Widget.WidgetSetFocusableMethodInfo
    ResolveDialogMethod "setFontMap" o = Gtk.Widget.WidgetSetFontMapMethodInfo
    ResolveDialogMethod "setFontOptions" o = Gtk.Widget.WidgetSetFontOptionsMethodInfo
    ResolveDialogMethod "setGravity" o = Gtk.Window.WindowSetGravityMethodInfo
    ResolveDialogMethod "setHalign" o = Gtk.Widget.WidgetSetHalignMethodInfo
    ResolveDialogMethod "setHandleMenubarAccel" o = Gtk.Window.WindowSetHandleMenubarAccelMethodInfo
    ResolveDialogMethod "setHasTooltip" o = Gtk.Widget.WidgetSetHasTooltipMethodInfo
    ResolveDialogMethod "setHexpand" o = Gtk.Widget.WidgetSetHexpandMethodInfo
    ResolveDialogMethod "setHexpandSet" o = Gtk.Widget.WidgetSetHexpandSetMethodInfo
    ResolveDialogMethod "setHideOnClose" o = Gtk.Window.WindowSetHideOnCloseMethodInfo
    ResolveDialogMethod "setIconName" o = Gtk.Window.WindowSetIconNameMethodInfo
    ResolveDialogMethod "setLayoutManager" o = Gtk.Widget.WidgetSetLayoutManagerMethodInfo
    ResolveDialogMethod "setLimitEvents" o = Gtk.Widget.WidgetSetLimitEventsMethodInfo
    ResolveDialogMethod "setMarginBottom" o = Gtk.Widget.WidgetSetMarginBottomMethodInfo
    ResolveDialogMethod "setMarginEnd" o = Gtk.Widget.WidgetSetMarginEndMethodInfo
    ResolveDialogMethod "setMarginStart" o = Gtk.Widget.WidgetSetMarginStartMethodInfo
    ResolveDialogMethod "setMarginTop" o = Gtk.Widget.WidgetSetMarginTopMethodInfo
    ResolveDialogMethod "setMnemonicsVisible" o = Gtk.Window.WindowSetMnemonicsVisibleMethodInfo
    ResolveDialogMethod "setModal" o = Gtk.Window.WindowSetModalMethodInfo
    ResolveDialogMethod "setName" o = Gtk.Widget.WidgetSetNameMethodInfo
    ResolveDialogMethod "setOpacity" o = Gtk.Widget.WidgetSetOpacityMethodInfo
    ResolveDialogMethod "setOverflow" o = Gtk.Widget.WidgetSetOverflowMethodInfo
    ResolveDialogMethod "setParent" o = Gtk.Widget.WidgetSetParentMethodInfo
    ResolveDialogMethod "setProperty" o = GObject.Object.ObjectSetPropertyMethodInfo
    ResolveDialogMethod "setReceivesDefault" o = Gtk.Widget.WidgetSetReceivesDefaultMethodInfo
    ResolveDialogMethod "setResizable" o = Gtk.Window.WindowSetResizableMethodInfo
    ResolveDialogMethod "setResponseSensitive" o = DialogSetResponseSensitiveMethodInfo
    ResolveDialogMethod "setSensitive" o = Gtk.Widget.WidgetSetSensitiveMethodInfo
    ResolveDialogMethod "setSizeRequest" o = Gtk.Widget.WidgetSetSizeRequestMethodInfo
    ResolveDialogMethod "setStartupId" o = Gtk.Window.WindowSetStartupIdMethodInfo
    ResolveDialogMethod "setStateFlags" o = Gtk.Widget.WidgetSetStateFlagsMethodInfo
    ResolveDialogMethod "setTitle" o = Gtk.Window.WindowSetTitleMethodInfo
    ResolveDialogMethod "setTitlebar" o = Gtk.Window.WindowSetTitlebarMethodInfo
    ResolveDialogMethod "setTooltipMarkup" o = Gtk.Widget.WidgetSetTooltipMarkupMethodInfo
    ResolveDialogMethod "setTooltipText" o = Gtk.Widget.WidgetSetTooltipTextMethodInfo
    ResolveDialogMethod "setTransientFor" o = Gtk.Window.WindowSetTransientForMethodInfo
    ResolveDialogMethod "setValign" o = Gtk.Widget.WidgetSetValignMethodInfo
    ResolveDialogMethod "setVexpand" o = Gtk.Widget.WidgetSetVexpandMethodInfo
    ResolveDialogMethod "setVexpandSet" o = Gtk.Widget.WidgetSetVexpandSetMethodInfo
    ResolveDialogMethod "setVisible" o = Gtk.Widget.WidgetSetVisibleMethodInfo
    ResolveDialogMethod l o = O.MethodResolutionFailed l o

instance (info ~ ResolveDialogMethod t Dialog, O.OverloadedMethod info Dialog p) => OL.IsLabel t (Dialog -> p) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.overloadedMethod @info
#else
    fromLabel _ = O.overloadedMethod @info
#endif

#if MIN_VERSION_base(4,13,0)
instance (info ~ ResolveDialogMethod t Dialog, O.OverloadedMethod info Dialog p, R.HasField t Dialog p) => R.HasField t Dialog p where
    getField = O.overloadedMethod @info

#endif

instance (info ~ ResolveDialogMethod t Dialog, O.OverloadedMethodInfo info Dialog) => OL.IsLabel t (O.MethodProxy info Dialog) where
#if MIN_VERSION_base(4,10,0)
    fromLabel = O.MethodProxy
#else
    fromLabel _ = O.MethodProxy
#endif

#endif

-- signal Dialog::close
{-# DEPRECATED DialogCloseCallback ["(Since version 4.10)","Use t'GI.Gtk.Objects.Window.Window' instead"] #-}
-- | Emitted when the user uses a keybinding to close the dialog.
-- 
-- This is a <https://docs.gtk.org/gtk4/class.SignalAction.html keybinding signal>.
-- 
-- The default binding for this signal is the Escape key.
type DialogCloseCallback =
    IO ()

type C_DialogCloseCallback =
    Ptr Dialog ->                           -- object
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_DialogCloseCallback`.
foreign import ccall "wrapper"
    mk_DialogCloseCallback :: C_DialogCloseCallback -> IO (FunPtr C_DialogCloseCallback)

wrap_DialogCloseCallback :: 
    GObject a => (a -> DialogCloseCallback) ->
    C_DialogCloseCallback
wrap_DialogCloseCallback gi'cb gi'selfPtr _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self) 


-- | Connect a signal handler for the [close](#signal:close) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' dialog #close callback
-- @
-- 
-- 
onDialogClose :: (IsDialog a, MonadIO m) => a -> ((?self :: a) => DialogCloseCallback) -> m SignalHandlerId
onDialogClose obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DialogCloseCallback wrapped
    wrapped'' <- mk_DialogCloseCallback wrapped'
    connectSignalFunPtr obj "close" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [close](#signal:close) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' dialog #close callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterDialogClose :: (IsDialog a, MonadIO m) => a -> ((?self :: a) => DialogCloseCallback) -> m SignalHandlerId
afterDialogClose obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DialogCloseCallback wrapped
    wrapped'' <- mk_DialogCloseCallback wrapped'
    connectSignalFunPtr obj "close" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data DialogCloseSignalInfo
instance SignalInfo DialogCloseSignalInfo where
    type HaskellCallbackType DialogCloseSignalInfo = DialogCloseCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_DialogCloseCallback cb
        cb'' <- mk_DialogCloseCallback cb'
        connectSignalFunPtr obj "close" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Dialog::close"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Dialog.html#g:signal:close"})

#endif

-- signal Dialog::response
{-# DEPRECATED DialogResponseCallback ["(Since version 4.10)","Use t'GI.Gtk.Objects.Window.Window' instead"] #-}
-- | Emitted when an action widget is clicked.
-- 
-- The signal is also emitted when the dialog receives a
-- delete event, and when 'GI.Gtk.Objects.Dialog.dialogResponse' is called.
-- On a delete event, the response ID is 'GI.Gtk.Enums.ResponseTypeDeleteEvent'.
-- Otherwise, it depends on which action widget was clicked.
type DialogResponseCallback =
    Int32
    -- ^ /@responseId@/: the response ID
    -> IO ()

type C_DialogResponseCallback =
    Ptr Dialog ->                           -- object
    Int32 ->
    Ptr () ->                               -- user_data
    IO ()

-- | Generate a function pointer callable from C code, from a `C_DialogResponseCallback`.
foreign import ccall "wrapper"
    mk_DialogResponseCallback :: C_DialogResponseCallback -> IO (FunPtr C_DialogResponseCallback)

wrap_DialogResponseCallback :: 
    GObject a => (a -> DialogResponseCallback) ->
    C_DialogResponseCallback
wrap_DialogResponseCallback gi'cb gi'selfPtr responseId _ = do
    B.ManagedPtr.withNewObject gi'selfPtr $ \gi'self -> gi'cb (Coerce.coerce gi'self)  responseId


-- | Connect a signal handler for the [response](#signal:response) signal, to be run before the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.on' dialog #response callback
-- @
-- 
-- 
onDialogResponse :: (IsDialog a, MonadIO m) => a -> ((?self :: a) => DialogResponseCallback) -> m SignalHandlerId
onDialogResponse obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DialogResponseCallback wrapped
    wrapped'' <- mk_DialogResponseCallback wrapped'
    connectSignalFunPtr obj "response" wrapped'' SignalConnectBefore Nothing

-- | Connect a signal handler for the [response](#signal:response) signal, to be run after the default handler.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Signals.after' dialog #response callback
-- @
-- 
-- 
-- 
-- By default the object invoking the signal is not passed to the callback.
-- If you need to access it, you can use the implit @?self@ parameter.
-- Note that this requires activating the @ImplicitParams@ GHC extension.
-- 
afterDialogResponse :: (IsDialog a, MonadIO m) => a -> ((?self :: a) => DialogResponseCallback) -> m SignalHandlerId
afterDialogResponse obj cb = liftIO $ do
    let wrapped self = let ?self = self in cb
    let wrapped' = wrap_DialogResponseCallback wrapped
    wrapped'' <- mk_DialogResponseCallback wrapped'
    connectSignalFunPtr obj "response" wrapped'' SignalConnectAfter Nothing


#if defined(ENABLE_OVERLOADING)
data DialogResponseSignalInfo
instance SignalInfo DialogResponseSignalInfo where
    type HaskellCallbackType DialogResponseSignalInfo = DialogResponseCallback
    connectSignal obj cb connectMode detail = do
        let cb' = wrap_DialogResponseCallback cb
        cb'' <- mk_DialogResponseCallback cb'
        connectSignalFunPtr obj "response" cb'' connectMode detail
    dbgSignalInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Dialog::response"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Dialog.html#g:signal:response"})

#endif

-- VVV Prop "use-header-bar"
   -- Type: TBasicType TInt
   -- Flags: [PropertyReadable,PropertyWritable,PropertyConstructOnly]
   -- Nullable: (Nothing,Nothing)

-- | Get the value of the “@use-header-bar@” property.
-- When <https://github.com/haskell-gi/haskell-gi/wiki/Overloading overloading> is enabled, this is equivalent to
-- 
-- @
-- 'Data.GI.Base.Attributes.get' dialog #useHeaderBar
-- @
getDialogUseHeaderBar :: (MonadIO m, IsDialog o) => o -> m Int32
getDialogUseHeaderBar obj = MIO.liftIO $ B.Properties.getObjectPropertyInt32 obj "use-header-bar"

-- | Construct a t'GValueConstruct' with valid value for the “@use-header-bar@” property. This is rarely needed directly, but it is used by `Data.GI.Base.Constructible.new`.
constructDialogUseHeaderBar :: (IsDialog o, MIO.MonadIO m) => Int32 -> m (GValueConstruct o)
constructDialogUseHeaderBar val = MIO.liftIO $ do
    MIO.liftIO $ B.Properties.constructObjectPropertyInt32 "use-header-bar" val

#if defined(ENABLE_OVERLOADING)
data DialogUseHeaderBarPropertyInfo
instance AttrInfo DialogUseHeaderBarPropertyInfo where
    type AttrAllowedOps DialogUseHeaderBarPropertyInfo = '[ 'AttrConstruct, 'AttrGet]
    type AttrBaseTypeConstraint DialogUseHeaderBarPropertyInfo = IsDialog
    type AttrSetTypeConstraint DialogUseHeaderBarPropertyInfo = (~) Int32
    type AttrTransferTypeConstraint DialogUseHeaderBarPropertyInfo = (~) Int32
    type AttrTransferType DialogUseHeaderBarPropertyInfo = Int32
    type AttrGetType DialogUseHeaderBarPropertyInfo = Int32
    type AttrLabel DialogUseHeaderBarPropertyInfo = "use-header-bar"
    type AttrOrigin DialogUseHeaderBarPropertyInfo = Dialog
    attrGet = getDialogUseHeaderBar
    attrSet = undefined
    attrTransfer _ v = do
        return v
    attrConstruct = constructDialogUseHeaderBar
    attrClear = undefined
    dbgAttrInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Dialog.useHeaderBar"
        , O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Dialog.html#g:attr:useHeaderBar"
        })
#endif

#if defined(ENABLE_OVERLOADING)
instance O.HasAttributeList Dialog
type instance O.AttributeList Dialog = DialogAttributeList
type DialogAttributeList = ('[ '("accessibleRole", Gtk.Accessible.AccessibleAccessibleRolePropertyInfo), '("application", Gtk.Window.WindowApplicationPropertyInfo), '("canFocus", Gtk.Widget.WidgetCanFocusPropertyInfo), '("canTarget", Gtk.Widget.WidgetCanTargetPropertyInfo), '("child", Gtk.Window.WindowChildPropertyInfo), '("cssClasses", Gtk.Widget.WidgetCssClassesPropertyInfo), '("cssName", Gtk.Widget.WidgetCssNamePropertyInfo), '("cursor", Gtk.Widget.WidgetCursorPropertyInfo), '("decorated", Gtk.Window.WindowDecoratedPropertyInfo), '("defaultHeight", Gtk.Window.WindowDefaultHeightPropertyInfo), '("defaultWidget", Gtk.Window.WindowDefaultWidgetPropertyInfo), '("defaultWidth", Gtk.Window.WindowDefaultWidthPropertyInfo), '("deletable", Gtk.Window.WindowDeletablePropertyInfo), '("destroyWithParent", Gtk.Window.WindowDestroyWithParentPropertyInfo), '("display", Gtk.Window.WindowDisplayPropertyInfo), '("focusOnClick", Gtk.Widget.WidgetFocusOnClickPropertyInfo), '("focusVisible", Gtk.Window.WindowFocusVisiblePropertyInfo), '("focusWidget", Gtk.Window.WindowFocusWidgetPropertyInfo), '("focusable", Gtk.Widget.WidgetFocusablePropertyInfo), '("fullscreened", Gtk.Window.WindowFullscreenedPropertyInfo), '("gravity", Gtk.Window.WindowGravityPropertyInfo), '("halign", Gtk.Widget.WidgetHalignPropertyInfo), '("handleMenubarAccel", Gtk.Window.WindowHandleMenubarAccelPropertyInfo), '("hasDefault", Gtk.Widget.WidgetHasDefaultPropertyInfo), '("hasFocus", Gtk.Widget.WidgetHasFocusPropertyInfo), '("hasTooltip", Gtk.Widget.WidgetHasTooltipPropertyInfo), '("heightRequest", Gtk.Widget.WidgetHeightRequestPropertyInfo), '("hexpand", Gtk.Widget.WidgetHexpandPropertyInfo), '("hexpandSet", Gtk.Widget.WidgetHexpandSetPropertyInfo), '("hideOnClose", Gtk.Window.WindowHideOnClosePropertyInfo), '("iconName", Gtk.Window.WindowIconNamePropertyInfo), '("isActive", Gtk.Window.WindowIsActivePropertyInfo), '("layoutManager", Gtk.Widget.WidgetLayoutManagerPropertyInfo), '("limitEvents", Gtk.Widget.WidgetLimitEventsPropertyInfo), '("marginBottom", Gtk.Widget.WidgetMarginBottomPropertyInfo), '("marginEnd", Gtk.Widget.WidgetMarginEndPropertyInfo), '("marginStart", Gtk.Widget.WidgetMarginStartPropertyInfo), '("marginTop", Gtk.Widget.WidgetMarginTopPropertyInfo), '("maximized", Gtk.Window.WindowMaximizedPropertyInfo), '("mnemonicsVisible", Gtk.Window.WindowMnemonicsVisiblePropertyInfo), '("modal", Gtk.Window.WindowModalPropertyInfo), '("name", Gtk.Widget.WidgetNamePropertyInfo), '("opacity", Gtk.Widget.WidgetOpacityPropertyInfo), '("overflow", Gtk.Widget.WidgetOverflowPropertyInfo), '("parent", Gtk.Widget.WidgetParentPropertyInfo), '("receivesDefault", Gtk.Widget.WidgetReceivesDefaultPropertyInfo), '("resizable", Gtk.Window.WindowResizablePropertyInfo), '("root", Gtk.Widget.WidgetRootPropertyInfo), '("scaleFactor", Gtk.Widget.WidgetScaleFactorPropertyInfo), '("sensitive", Gtk.Widget.WidgetSensitivePropertyInfo), '("startupId", Gtk.Window.WindowStartupIdPropertyInfo), '("suspended", Gtk.Window.WindowSuspendedPropertyInfo), '("title", Gtk.Window.WindowTitlePropertyInfo), '("titlebar", Gtk.Window.WindowTitlebarPropertyInfo), '("tooltipMarkup", Gtk.Widget.WidgetTooltipMarkupPropertyInfo), '("tooltipText", Gtk.Widget.WidgetTooltipTextPropertyInfo), '("transientFor", Gtk.Window.WindowTransientForPropertyInfo), '("useHeaderBar", DialogUseHeaderBarPropertyInfo), '("valign", Gtk.Widget.WidgetValignPropertyInfo), '("vexpand", Gtk.Widget.WidgetVexpandPropertyInfo), '("vexpandSet", Gtk.Widget.WidgetVexpandSetPropertyInfo), '("visible", Gtk.Widget.WidgetVisiblePropertyInfo), '("widthRequest", Gtk.Widget.WidgetWidthRequestPropertyInfo)] :: [(Symbol, DK.Type)])
#endif

#if defined(ENABLE_OVERLOADING)
dialogUseHeaderBar :: AttrLabelProxy "useHeaderBar"
dialogUseHeaderBar = AttrLabelProxy

#endif

#if defined(ENABLE_OVERLOADING)
type instance O.SignalList Dialog = DialogSignalList
type DialogSignalList = ('[ '("activateDefault", Gtk.Window.WindowActivateDefaultSignalInfo), '("activateFocus", Gtk.Window.WindowActivateFocusSignalInfo), '("close", DialogCloseSignalInfo), '("closeRequest", Gtk.Window.WindowCloseRequestSignalInfo), '("destroy", Gtk.Widget.WidgetDestroySignalInfo), '("directionChanged", Gtk.Widget.WidgetDirectionChangedSignalInfo), '("enableDebugging", Gtk.Window.WindowEnableDebuggingSignalInfo), '("hide", Gtk.Widget.WidgetHideSignalInfo), '("keynavFailed", Gtk.Widget.WidgetKeynavFailedSignalInfo), '("keysChanged", Gtk.Window.WindowKeysChangedSignalInfo), '("map", Gtk.Widget.WidgetMapSignalInfo), '("mnemonicActivate", Gtk.Widget.WidgetMnemonicActivateSignalInfo), '("moveFocus", Gtk.Widget.WidgetMoveFocusSignalInfo), '("notify", GObject.Object.ObjectNotifySignalInfo), '("queryTooltip", Gtk.Widget.WidgetQueryTooltipSignalInfo), '("realize", Gtk.Widget.WidgetRealizeSignalInfo), '("response", DialogResponseSignalInfo), '("show", Gtk.Widget.WidgetShowSignalInfo), '("stateFlagsChanged", Gtk.Widget.WidgetStateFlagsChangedSignalInfo), '("unmap", Gtk.Widget.WidgetUnmapSignalInfo), '("unrealize", Gtk.Widget.WidgetUnrealizeSignalInfo)] :: [(Symbol, DK.Type)])

#endif

-- method Dialog::new
-- method type : Constructor
-- Args: []
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Dialog" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_dialog_new" gtk_dialog_new :: 
    IO (Ptr Dialog)

{-# DEPRECATED dialogNew ["(Since version 4.10)","Use t'GI.Gtk.Objects.Window.Window' instead"] #-}
-- | Creates a new dialog box.
-- 
-- Widgets should not be packed into the @GtkWindow@
-- directly, but into the /@contentArea@/ and /@actionArea@/,
-- as described above.
dialogNew ::
    (B.CallStack.HasCallStack, MonadIO m) =>
    m Dialog
    -- ^ __Returns:__ the new dialog as a @GtkWidget@
dialogNew  = liftIO $ do
    result <- gtk_dialog_new
    checkUnexpectedReturnNULL "dialogNew" result
    result' <- (newObject Dialog) result
    return result'

#if defined(ENABLE_OVERLOADING)
#endif

-- method Dialog::add_action_widget
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dialog"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Dialog" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDialog`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "child"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "an activatable widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "response_id"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "response ID for @child"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_dialog_add_action_widget" gtk_dialog_add_action_widget :: 
    Ptr Dialog ->                           -- dialog : TInterface (Name {namespace = "Gtk", name = "Dialog"})
    Ptr Gtk.Widget.Widget ->                -- child : TInterface (Name {namespace = "Gtk", name = "Widget"})
    Int32 ->                                -- response_id : TBasicType TInt
    IO ()

{-# DEPRECATED dialogAddActionWidget ["(Since version 4.10)","Use t'GI.Gtk.Objects.Window.Window' instead"] #-}
-- | Adds an activatable widget to the action area of a @GtkDialog@.
-- 
-- GTK connects a signal handler that will emit the
-- [Dialog::response]("GI.Gtk.Objects.Dialog#g:signal:response") signal on the dialog when the widget
-- is activated. The widget is appended to the end of the dialog’s action
-- area.
-- 
-- If you want to add a non-activatable widget, simply pack it into
-- the /@actionArea@/ field of the @GtkDialog@ struct.
dialogAddActionWidget ::
    (B.CallStack.HasCallStack, MonadIO m, IsDialog a, Gtk.Widget.IsWidget b) =>
    a
    -- ^ /@dialog@/: a @GtkDialog@
    -> b
    -- ^ /@child@/: an activatable widget
    -> Int32
    -- ^ /@responseId@/: response ID for /@child@/
    -> m ()
dialogAddActionWidget dialog child responseId = liftIO $ do
    dialog' <- unsafeManagedPtrCastPtr dialog
    child' <- unsafeManagedPtrCastPtr child
    gtk_dialog_add_action_widget dialog' child' responseId
    touchManagedPtr dialog
    touchManagedPtr child
    return ()

#if defined(ENABLE_OVERLOADING)
data DialogAddActionWidgetMethodInfo
instance (signature ~ (b -> Int32 -> m ()), MonadIO m, IsDialog a, Gtk.Widget.IsWidget b) => O.OverloadedMethod DialogAddActionWidgetMethodInfo a signature where
    overloadedMethod = dialogAddActionWidget

instance O.OverloadedMethodInfo DialogAddActionWidgetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Dialog.dialogAddActionWidget",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Dialog.html#v:dialogAddActionWidget"
        })


#endif

-- method Dialog::add_button
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dialog"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Dialog" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDialog`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "button_text"
--           , argType = TBasicType TUTF8
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "text of button" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "response_id"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "response ID for the button"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Widget" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_dialog_add_button" gtk_dialog_add_button :: 
    Ptr Dialog ->                           -- dialog : TInterface (Name {namespace = "Gtk", name = "Dialog"})
    CString ->                              -- button_text : TBasicType TUTF8
    Int32 ->                                -- response_id : TBasicType TInt
    IO (Ptr Gtk.Widget.Widget)

{-# DEPRECATED dialogAddButton ["(Since version 4.10)","Use t'GI.Gtk.Objects.Window.Window' instead"] #-}
-- | Adds a button with the given text.
-- 
-- GTK arranges things so that clicking the button will emit the
-- [Dialog::response]("GI.Gtk.Objects.Dialog#g:signal:response") signal with the given /@responseId@/.
-- The button is appended to the end of the dialog’s action area.
-- The button widget is returned, but usually you don’t need it.
dialogAddButton ::
    (B.CallStack.HasCallStack, MonadIO m, IsDialog a) =>
    a
    -- ^ /@dialog@/: a @GtkDialog@
    -> T.Text
    -- ^ /@buttonText@/: text of button
    -> Int32
    -- ^ /@responseId@/: response ID for the button
    -> m Gtk.Widget.Widget
    -- ^ __Returns:__ the @GtkButton@ widget that was added
dialogAddButton dialog buttonText responseId = liftIO $ do
    dialog' <- unsafeManagedPtrCastPtr dialog
    buttonText' <- textToCString buttonText
    result <- gtk_dialog_add_button dialog' buttonText' responseId
    checkUnexpectedReturnNULL "dialogAddButton" result
    result' <- (newObject Gtk.Widget.Widget) result
    touchManagedPtr dialog
    freeMem buttonText'
    return result'

#if defined(ENABLE_OVERLOADING)
data DialogAddButtonMethodInfo
instance (signature ~ (T.Text -> Int32 -> m Gtk.Widget.Widget), MonadIO m, IsDialog a) => O.OverloadedMethod DialogAddButtonMethodInfo a signature where
    overloadedMethod = dialogAddButton

instance O.OverloadedMethodInfo DialogAddButtonMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Dialog.dialogAddButton",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Dialog.html#v:dialogAddButton"
        })


#endif

-- method Dialog::get_content_area
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dialog"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Dialog" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDialog`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Box" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_dialog_get_content_area" gtk_dialog_get_content_area :: 
    Ptr Dialog ->                           -- dialog : TInterface (Name {namespace = "Gtk", name = "Dialog"})
    IO (Ptr Gtk.Box.Box)

{-# DEPRECATED dialogGetContentArea ["(Since version 4.10)","Use t'GI.Gtk.Objects.Window.Window' instead"] #-}
-- | Returns the content area of /@dialog@/.
dialogGetContentArea ::
    (B.CallStack.HasCallStack, MonadIO m, IsDialog a) =>
    a
    -- ^ /@dialog@/: a @GtkDialog@
    -> m Gtk.Box.Box
    -- ^ __Returns:__ the content area @GtkBox@.
dialogGetContentArea dialog = liftIO $ do
    dialog' <- unsafeManagedPtrCastPtr dialog
    result <- gtk_dialog_get_content_area dialog'
    checkUnexpectedReturnNULL "dialogGetContentArea" result
    result' <- (newObject Gtk.Box.Box) result
    touchManagedPtr dialog
    return result'

#if defined(ENABLE_OVERLOADING)
data DialogGetContentAreaMethodInfo
instance (signature ~ (m Gtk.Box.Box), MonadIO m, IsDialog a) => O.OverloadedMethod DialogGetContentAreaMethodInfo a signature where
    overloadedMethod = dialogGetContentArea

instance O.OverloadedMethodInfo DialogGetContentAreaMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Dialog.dialogGetContentArea",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Dialog.html#v:dialogGetContentArea"
        })


#endif

-- method Dialog::get_header_bar
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dialog"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Dialog" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDialog`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "HeaderBar" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_dialog_get_header_bar" gtk_dialog_get_header_bar :: 
    Ptr Dialog ->                           -- dialog : TInterface (Name {namespace = "Gtk", name = "Dialog"})
    IO (Ptr Gtk.HeaderBar.HeaderBar)

{-# DEPRECATED dialogGetHeaderBar ["(Since version 4.10)","Use t'GI.Gtk.Objects.Window.Window' instead"] #-}
-- | Returns the header bar of /@dialog@/.
-- 
-- Note that the headerbar is only used by the dialog if the
-- [Dialog:useHeaderBar]("GI.Gtk.Objects.Dialog#g:attr:useHeaderBar") property is 'P.True'.
dialogGetHeaderBar ::
    (B.CallStack.HasCallStack, MonadIO m, IsDialog a) =>
    a
    -- ^ /@dialog@/: a @GtkDialog@
    -> m Gtk.HeaderBar.HeaderBar
    -- ^ __Returns:__ the header bar
dialogGetHeaderBar dialog = liftIO $ do
    dialog' <- unsafeManagedPtrCastPtr dialog
    result <- gtk_dialog_get_header_bar dialog'
    checkUnexpectedReturnNULL "dialogGetHeaderBar" result
    result' <- (newObject Gtk.HeaderBar.HeaderBar) result
    touchManagedPtr dialog
    return result'

#if defined(ENABLE_OVERLOADING)
data DialogGetHeaderBarMethodInfo
instance (signature ~ (m Gtk.HeaderBar.HeaderBar), MonadIO m, IsDialog a) => O.OverloadedMethod DialogGetHeaderBarMethodInfo a signature where
    overloadedMethod = dialogGetHeaderBar

instance O.OverloadedMethodInfo DialogGetHeaderBarMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Dialog.dialogGetHeaderBar",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Dialog.html#v:dialogGetHeaderBar"
        })


#endif

-- method Dialog::get_response_for_widget
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dialog"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Dialog" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDialog`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "widget"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Widget" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a widget in the action area of @dialog"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TBasicType TInt)
-- throws : False
-- Skip return : False

foreign import ccall "gtk_dialog_get_response_for_widget" gtk_dialog_get_response_for_widget :: 
    Ptr Dialog ->                           -- dialog : TInterface (Name {namespace = "Gtk", name = "Dialog"})
    Ptr Gtk.Widget.Widget ->                -- widget : TInterface (Name {namespace = "Gtk", name = "Widget"})
    IO Int32

{-# DEPRECATED dialogGetResponseForWidget ["(Since version 4.10)","Use t'GI.Gtk.Objects.Window.Window' instead"] #-}
-- | Gets the response id of a widget in the action area
-- of a dialog.
dialogGetResponseForWidget ::
    (B.CallStack.HasCallStack, MonadIO m, IsDialog a, Gtk.Widget.IsWidget b) =>
    a
    -- ^ /@dialog@/: a @GtkDialog@
    -> b
    -- ^ /@widget@/: a widget in the action area of /@dialog@/
    -> m Int32
    -- ^ __Returns:__ the response id of /@widget@/, or 'GI.Gtk.Enums.ResponseTypeNone'
    --  if /@widget@/ doesn’t have a response id set.
dialogGetResponseForWidget dialog widget = liftIO $ do
    dialog' <- unsafeManagedPtrCastPtr dialog
    widget' <- unsafeManagedPtrCastPtr widget
    result <- gtk_dialog_get_response_for_widget dialog' widget'
    touchManagedPtr dialog
    touchManagedPtr widget
    return result

#if defined(ENABLE_OVERLOADING)
data DialogGetResponseForWidgetMethodInfo
instance (signature ~ (b -> m Int32), MonadIO m, IsDialog a, Gtk.Widget.IsWidget b) => O.OverloadedMethod DialogGetResponseForWidgetMethodInfo a signature where
    overloadedMethod = dialogGetResponseForWidget

instance O.OverloadedMethodInfo DialogGetResponseForWidgetMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Dialog.dialogGetResponseForWidget",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Dialog.html#v:dialogGetResponseForWidget"
        })


#endif

-- method Dialog::get_widget_for_response
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dialog"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Dialog" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDialog`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "response_id"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "the response ID used by the @dialog widget"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Just (TInterface Name { namespace = "Gtk" , name = "Widget" })
-- throws : False
-- Skip return : False

foreign import ccall "gtk_dialog_get_widget_for_response" gtk_dialog_get_widget_for_response :: 
    Ptr Dialog ->                           -- dialog : TInterface (Name {namespace = "Gtk", name = "Dialog"})
    Int32 ->                                -- response_id : TBasicType TInt
    IO (Ptr Gtk.Widget.Widget)

{-# DEPRECATED dialogGetWidgetForResponse ["(Since version 4.10)","Use t'GI.Gtk.Objects.Window.Window' instead"] #-}
-- | Gets the widget button that uses the given response ID in the action area
-- of a dialog.
dialogGetWidgetForResponse ::
    (B.CallStack.HasCallStack, MonadIO m, IsDialog a) =>
    a
    -- ^ /@dialog@/: a @GtkDialog@
    -> Int32
    -- ^ /@responseId@/: the response ID used by the /@dialog@/ widget
    -> m (Maybe Gtk.Widget.Widget)
    -- ^ __Returns:__ the /@widget@/ button that uses the given
    --   /@responseId@/
dialogGetWidgetForResponse dialog responseId = liftIO $ do
    dialog' <- unsafeManagedPtrCastPtr dialog
    result <- gtk_dialog_get_widget_for_response dialog' responseId
    maybeResult <- convertIfNonNull result $ \result' -> do
        result'' <- (newObject Gtk.Widget.Widget) result'
        return result''
    touchManagedPtr dialog
    return maybeResult

#if defined(ENABLE_OVERLOADING)
data DialogGetWidgetForResponseMethodInfo
instance (signature ~ (Int32 -> m (Maybe Gtk.Widget.Widget)), MonadIO m, IsDialog a) => O.OverloadedMethod DialogGetWidgetForResponseMethodInfo a signature where
    overloadedMethod = dialogGetWidgetForResponse

instance O.OverloadedMethodInfo DialogGetWidgetForResponseMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Dialog.dialogGetWidgetForResponse",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Dialog.html#v:dialogGetWidgetForResponse"
        })


#endif

-- method Dialog::response
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dialog"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Dialog" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDialog`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "response_id"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "response ID" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_dialog_response" gtk_dialog_response :: 
    Ptr Dialog ->                           -- dialog : TInterface (Name {namespace = "Gtk", name = "Dialog"})
    Int32 ->                                -- response_id : TBasicType TInt
    IO ()

{-# DEPRECATED dialogResponse ["(Since version 4.10)","Use t'GI.Gtk.Objects.Window.Window' instead"] #-}
-- | Emits the [response](#g:signal:response) signal with the given response ID.
-- 
-- Used to indicate that the user has responded to the dialog in some way.
dialogResponse ::
    (B.CallStack.HasCallStack, MonadIO m, IsDialog a) =>
    a
    -- ^ /@dialog@/: a @GtkDialog@
    -> Int32
    -- ^ /@responseId@/: response ID
    -> m ()
dialogResponse dialog responseId = liftIO $ do
    dialog' <- unsafeManagedPtrCastPtr dialog
    gtk_dialog_response dialog' responseId
    touchManagedPtr dialog
    return ()

#if defined(ENABLE_OVERLOADING)
data DialogResponseMethodInfo
instance (signature ~ (Int32 -> m ()), MonadIO m, IsDialog a) => O.OverloadedMethod DialogResponseMethodInfo a signature where
    overloadedMethod = dialogResponse

instance O.OverloadedMethodInfo DialogResponseMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Dialog.dialogResponse",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Dialog.html#v:dialogResponse"
        })


#endif

-- method Dialog::set_default_response
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dialog"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Dialog" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDialog`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "response_id"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a response ID" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_dialog_set_default_response" gtk_dialog_set_default_response :: 
    Ptr Dialog ->                           -- dialog : TInterface (Name {namespace = "Gtk", name = "Dialog"})
    Int32 ->                                -- response_id : TBasicType TInt
    IO ()

{-# DEPRECATED dialogSetDefaultResponse ["(Since version 4.10)","Use t'GI.Gtk.Objects.Window.Window' instead"] #-}
-- | Sets the default widget for the dialog based on the response ID.
-- 
-- Pressing “Enter” normally activates the default widget.
dialogSetDefaultResponse ::
    (B.CallStack.HasCallStack, MonadIO m, IsDialog a) =>
    a
    -- ^ /@dialog@/: a @GtkDialog@
    -> Int32
    -- ^ /@responseId@/: a response ID
    -> m ()
dialogSetDefaultResponse dialog responseId = liftIO $ do
    dialog' <- unsafeManagedPtrCastPtr dialog
    gtk_dialog_set_default_response dialog' responseId
    touchManagedPtr dialog
    return ()

#if defined(ENABLE_OVERLOADING)
data DialogSetDefaultResponseMethodInfo
instance (signature ~ (Int32 -> m ()), MonadIO m, IsDialog a) => O.OverloadedMethod DialogSetDefaultResponseMethodInfo a signature where
    overloadedMethod = dialogSetDefaultResponse

instance O.OverloadedMethodInfo DialogSetDefaultResponseMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Dialog.dialogSetDefaultResponse",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Dialog.html#v:dialogSetDefaultResponse"
        })


#endif

-- method Dialog::set_response_sensitive
-- method type : OrdinaryMethod
-- Args: [ Arg
--           { argCName = "dialog"
--           , argType = TInterface Name { namespace = "Gtk" , name = "Dialog" }
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a `GtkDialog`" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "response_id"
--           , argType = TBasicType TInt
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "a response ID" , sinceVersion = Nothing }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       , Arg
--           { argCName = "setting"
--           , argType = TBasicType TBoolean
--           , direction = DirectionIn
--           , mayBeNull = False
--           , argDoc =
--               Documentation
--                 { rawDocText = Just "%TRUE for sensitive"
--                 , sinceVersion = Nothing
--                 }
--           , argScope = ScopeTypeInvalid
--           , argClosure = -1
--           , argDestroy = -1
--           , argCallerAllocates = False
--           , argCallbackUserData = False
--           , transfer = TransferNothing
--           }
--       ]
-- Lengths: []
-- returnType: Nothing
-- throws : False
-- Skip return : False

foreign import ccall "gtk_dialog_set_response_sensitive" gtk_dialog_set_response_sensitive :: 
    Ptr Dialog ->                           -- dialog : TInterface (Name {namespace = "Gtk", name = "Dialog"})
    Int32 ->                                -- response_id : TBasicType TInt
    CInt ->                                 -- setting : TBasicType TBoolean
    IO ()

{-# DEPRECATED dialogSetResponseSensitive ["(Since version 4.10)","Use t'GI.Gtk.Objects.Window.Window' instead"] #-}
-- | A convenient way to sensitize\/desensitize dialog buttons.
-- 
-- Calls @gtk_widget_set_sensitive (widget, \@setting)@
-- for each widget in the dialog’s action area with the given /@responseId@/.
dialogSetResponseSensitive ::
    (B.CallStack.HasCallStack, MonadIO m, IsDialog a) =>
    a
    -- ^ /@dialog@/: a @GtkDialog@
    -> Int32
    -- ^ /@responseId@/: a response ID
    -> Bool
    -- ^ /@setting@/: 'P.True' for sensitive
    -> m ()
dialogSetResponseSensitive dialog responseId setting = liftIO $ do
    dialog' <- unsafeManagedPtrCastPtr dialog
    let setting' = (P.fromIntegral . P.fromEnum) setting
    gtk_dialog_set_response_sensitive dialog' responseId setting'
    touchManagedPtr dialog
    return ()

#if defined(ENABLE_OVERLOADING)
data DialogSetResponseSensitiveMethodInfo
instance (signature ~ (Int32 -> Bool -> m ()), MonadIO m, IsDialog a) => O.OverloadedMethod DialogSetResponseSensitiveMethodInfo a signature where
    overloadedMethod = dialogSetResponseSensitive

instance O.OverloadedMethodInfo DialogSetResponseSensitiveMethodInfo a where
    overloadedMethodInfo = P.Just (O.ResolvedSymbolInfo {
        O.resolvedSymbolName = "GI.Gtk.Objects.Dialog.dialogSetResponseSensitive",
        O.resolvedSymbolURL = "https://hackage.haskell.org/package/gi-gtk-4.0.11/docs/GI-Gtk-Objects-Dialog.html#v:dialogSetResponseSensitive"
        })


#endif


